;; Wojciech Jedynak (wjedynak@gmail.com)

;; Bonus: In this file we demonstrate how to encode various semantic
;; formats in PLT Redex using Hutton's Razor as the running example.

(require redex)

;;-------------------------------------------------------------------
;; Part 01: The language (A for arith)
;;-------------------------------------------------------------------

;; Informal grammar: e ::= n | e + e

(define-language A
  (e n
     (+ e e))
  (n number))

;;-------------------------------------------------------------------
;; Part 02: Reduction Semantics
;;-------------------------------------------------------------------

;; The semantic format best (natively) supported in Redex

;; Evaluation contexts

(define-extended-language AR A
  (E hole
     (+ E e)
     (+ n E))
  (r (+ n_0 n_1)))                      ; redex

(define-metafunction AR
  Σ : n n -> n
  [(Σ n_0 n_1)
   ,(+ (term n_0) (term n_1))])

(define rs-red
  (reduction-relation
   AR
   (--> (in-hole E (+ n_0 n_1))
        (in-hole E (Σ n_0 n_1)))))

;; Examples

(define (example-traces)
  (traces rs-red (term (+ 1 2)))
  (traces rs-red (term (+ (+ 1 1) 2)))
  (traces rs-red (term (+ 1 (+ 2 (+ 3 4))))))

;; (example-traces)

;; Unit tests
(define (test-red-sem)
  (test-->> rs-red
   (term (+ 1 2))
   (term 3))

  (test-->>
   rs-red
   (term (+ 1 (+ 2 (+ 3 4))))
   (term 10))

  (test-results))

;; (test-red-sem)

;; Desired properties

;; Property 01: Unique decomposition

(define (value? e)
  (redex-match AR n e))

(define (unique-decomposition? e)
  (or (value? e)
      (= 1 (length (redex-match AR (in-hole E r) e)))))

(define (prop-red-sem)
  (redex-check AR e (unique-decomposition? (term e))))

;; (prop-red-sem)

;;-------------------------------------------------------------------
;; Part 03: Abstract Machine
;;-------------------------------------------------------------------

;; This format is also suitable for PLT Redex, because it's
;; effectively a transition system.

(define-extended-language AM A
  (s stop                               ; stacks
     (arg-1 e s)
     (arg-2 n s))
  (c (ev e s)                           ; configurations
     (ap s n)))

(define am-red
  (reduction-relation
   AM
   (--> (ev n s)
        (ap s n))
   (--> (ev (+ e_0 e_1) s)
        (ev e_0 (arg-1 e_1 s)))
   (--> (ap (arg-1 e_1 s) n_0)
        (ev e_1 (arg-2 n_0 s)))
   (--> (ap (arg-2 n_0 s) n_1)
        (ap s (Σ n_0 n_1)))))

;; We can highlight states that don't satisfy a condition!

;; The condition
(define (eval-state? c)
  (redex-match AM (ev e s) c))

(define (am-traces)
  (traces am-red (term (ev (+ 1 2) stop)))
  (traces am-red (term (ev (+ (+ (+ 1 2) 3) 4) stop)))
  ;; This will highlight non-eval states
  (traces am-red (term (ev (+ (+ (+ 1 2) 3) 4) stop))
          #:pred eval-state?)
  )

;; (am-traces)

(define (test-am-red)
  (test-->>
   am-red
   (term (ev (+ 1 2) stop))
   (term (ap stop 3)))

  (test-results))

;; (test-am-red)

;; Desired properties

;; Property 01: equivalence with reduction semantics

(define-metafunction AM
  get-result : c -> n
  [(get-result (ap stop n)) n])

(define (equivalent-rs-am? e verbose?)
  (let [(rs (apply-reduction-relation* rs-red e))
        (am (apply-reduction-relation* am-red (term (ev ,e stop))))]
    (when verbose?
        (printf "~s\n" e))
    (and (= 1 (length rs))
         (= 1 (length am))
         (equal? (first rs)
                 (term (get-result ,(first am)))))))

(define (rs-am-random-checks)
  (redex-check A e (equivalent-rs-am? (term e) #f)) ; #t to print arg

  ;; possibly better coverage
  (redex-check AR e (equivalent-rs-am? (term e) #f)
               #:source rs-red))

;; (rs-am-random-checks)

;;-------------------------------------------------------------------
;; Part 04: Structural Operational Semantics (SOS)
;;-------------------------------------------------------------------

(define-judgment-form A
  #:mode     (sos I O)
  #:contract (sos e e)

  [-------------------------------- ;; SOS-plus
   (sos (+ n_0 n_1) (Σ n_0 n_1))]

  [(sos e_0 e_1)
   -------------------------------- ;; SOS-plus-left
   (sos (+ e_0 e) (+ e_1 e))]

  [(sos e_0 e_1)
   -------------------------------- ;; SOS-plus-right
   (sos (+ n e_0) (+ n e_1))]
)

;; Testing SOS -- Redex can indeed use the judgment as a function!

(judgment-holds
 (sos (+ 1 2) e)
 e)

(judgment-holds
 (sos (+ (+ 0 1) 2) e)
 e)

(judgment-holds
 (sos (+ 2 (+ 0 1)) e)
 e)

(define (sos-tests)
  (test-equal
   (judgment-holds (sos (+ 1 2) e) e)
   (list (term 3)))
  (test-equal
   (judgment-holds (sos (+ 2 (+ 0 1)) e) e)
   (list (term (+ 2 1))))
  (test-results))

;; (sos-tests)

;; Transitive closure

(define (sos-trans e)
  ;; (print e)
  ;; (newline)
  (if (value? e)
      e
      (sos-trans (car (judgment-holds (sos ,e e_0) e_0)))))

(sos-trans (term (+ (+ 1 2) (+ 1 2))))

;; Desired properties

;; Property 01: equivalence with reduction semantics

(define (prop-sos)
  (redex-check
   A e
   (equal? (first (apply-reduction-relation* rs-red (term e)))
           (sos-trans (term e)))))

;; (prop-sos)

;;-------------------------------------------------------------------
;; Part 05: Natural Semantics (NS)
;;-------------------------------------------------------------------

(define-judgment-form A
  #:mode     (ns I O)
  #:contract (ns e n)

  [--------- ;; NS-LIT
   (ns n n)]

  [(ns e_0 n_0)
   (ns e_1 n_1)
   ------------ ;; NS-PLUS
   (ns (+ e_0 e_1) (Σ n_0 n_1))]
)

;; tests

(judgment-holds (ns (+ 1 2) n) n)

(judgment-holds (ns (+ (+ 1 2) (+ 3 4)) n) n)

;; Desired properties

(define (prop-ns)

  ;; Property 01: natural semantics is deterministic

  (redex-check
   A e
   (= 1 (length (judgment-holds (ns e n) n))))

  ;; Property 02: natural semantics is equivalent to reduction semantics

  (redex-check
   A e
   (equal? (first (judgment-holds (ns e n) n))
           (first (apply-reduction-relation* rs-red (term e))))))

;; (prop-ns)


