using System;
using System.Drawing;
using System.Windows.Forms;

using Vulcan.Uczniowie.Win32;

namespace Vulcan.Uczniowie.Controls
{
	internal class ToolTip : IDisposable
	{
		#region Pola, konstruktory

		// Tooltip Window Constants
		const short  TTDT_AUTOPOP = 2;
		const short  TTDT_INITIAL = 3;
		const string TOOLTIPS_CLASSA = "tooltips_class32";

		// local variable(s) to hold property value(s)
		string m_Title;
		string m_TipText;
		int    m_MaxTipWidth;
		int    m_BackColor;
		int    m_ForeColor;
		int    m_DelayTime;
		int    m_VisibleTime;
		bool   m_Centered;
		bool   m_PopupOnDemand;

		ToolTipStyle    m_Style;
		ToolTipIconType m_Icon;

		// private data
		IntPtr m_lTTHwnd;     // hwnd of the tooltip
		IntPtr m_lParentHwnd; // hwnd of the window the tooltip attached to
		Font   m_TipFont;
		TOOLINFO ti;

		public ToolTip()
		{			
			WindowsAPI.InitCommonControls();
			m_MaxTipWidth   = 800; 
			m_DelayTime     = 500;
			m_VisibleTime   = 5000;
			m_PopupOnDemand = false;
		}

		/*
		~WZToolTip()
		{
			Dispose();
		}
		*/
		#endregion

		#region Properties
		public int MaxTipWidth
		{
			get
			{			
				return m_MaxTipWidth;
			}
			set 
			{
				m_MaxTipWidth = value;
				if ( m_lTTHwnd != IntPtr.Zero )
				{
					WindowsAPI.SendMessage( m_lTTHwnd, (int)ToolTipMessages.TTM_SETMAXTIPWIDTH, 0, m_MaxTipWidth );
				}
			}
		}
		
		public int VisibleTime
		{
			get
			{			
				return m_VisibleTime;
			}
			set 
			{
				m_VisibleTime = value;
				if ( m_lTTHwnd != IntPtr.Zero )
				{
					WindowsAPI.SendMessage( m_lTTHwnd, (int)ToolTipMessages.TTM_SETDELAYTIME, TTDT_AUTOPOP, m_VisibleTime );
				}
			}
		}
		
		public int DelayTime
		{
			get
			{
				return m_DelayTime;
			}
			set 
			{
				m_DelayTime = value;
				if ( m_lTTHwnd != IntPtr.Zero )
				{
					WindowsAPI.SendMessage( m_lTTHwnd, (int)ToolTipMessages.TTM_SETDELAYTIME, TTDT_INITIAL, m_DelayTime );
				}
			}
		}
		
		public Font Font
		{
			get
			{
				return m_TipFont;
			}
			set 
			{
				m_TipFont = value;
				if ( m_lTTHwnd != IntPtr.Zero )
				{
					WindowsAPI.SendMessage( m_lTTHwnd, (int)Msg.WM_SETFONT, m_TipFont.ToHfont().ToInt32(), 1 );
				}
			}
		}
		
		public ToolTipIconType Icon
		{
			get
			{
				return m_Icon;
			}
			set 
			{
				m_Icon = value;
				if ( m_lTTHwnd != IntPtr.Zero &&
					 m_Title != string.Empty &&
                     m_Icon != ToolTipIconType.TT_NOICON
					) 
				{
					WindowsAPI.SendMessage( m_lTTHwnd, (int)ToolTipMessages.TTM_SETTITLE, (int)m_Icon, m_Title );
				}
			}
		}

		public ToolTipStyle Style
		{
			get
			{
				return m_Style;
			}
		}

		public int BackColor
		{
			get
			{
				return m_BackColor;
			}			
			set
			{
				m_BackColor = value;
				if ( m_lTTHwnd != IntPtr.Zero )
				{
					WindowsAPI.SendMessage( m_lTTHwnd, (int)ToolTipMessages.TTM_SETTIPBKCOLOR, m_BackColor, 0 );
				}
		
			}
		}
		
		public int ForeColor
		{
			get
			{
				return m_ForeColor;
			}
			set 
			{
				m_ForeColor = value;
				if ( m_lTTHwnd != IntPtr.Zero )
				{
					WindowsAPI.SendMessage( m_lTTHwnd, (int)ToolTipMessages.TTM_SETTIPTEXTCOLOR, m_ForeColor, 0 );
				}
			}
		}
		
		public string Title 
		{
			get
			{			
				return ti.lpStr;
			}
			set 
			{
				m_Title = value;
				if ( m_lTTHwnd != IntPtr.Zero &&
					m_Title != null &&
					m_Icon != ToolTipIconType.TT_NOICON
					)
				{
					WindowsAPI.SendMessage( m_lTTHwnd, (int)ToolTipMessages.TTM_SETTITLE, (int)m_Icon, m_Title );
				}
			}
		}
		
		public string ToolTipText
		{
			get
			{
				return m_TipText;
			}
			set
			{
				m_TipText = value;
				ti.lpStr  = value;
				if ( m_lTTHwnd != IntPtr.Zero )
				{
					WindowsAPI.SendMessage( m_lTTHwnd, (int)ToolTipMessages.TTM_UPDATETIPTEXTA, 0, ref ti );
				}
			}
		}
		
		public bool PopupOnDemand
		{
			get
			{
				return m_PopupOnDemand;
			}
			set
			{
				m_PopupOnDemand = value;
			}
		}
		
		public bool Centered
		{
			get
			{
				return m_Centered;
			}
			set
			{
				m_Centered = value;
			}
		}
		#endregion

		#region Methods
		public void Hide()
		{
			if ( m_lTTHwnd != IntPtr.Zero )
			{
				WindowsAPI.SendMessage( m_lTTHwnd, (int)ToolTipMessages.TTM_TRACKACTIVATE, 0, ref ti );
			}
		}	

		public void Show( int X, int Y )
		{
			if ( m_lTTHwnd != IntPtr.Zero )
			{
				POINT pt;
				pt.x = X;
				pt.y = Y;

				WindowsAPI.ClientToScreen( m_lParentHwnd, ref pt );

				int ptTip;
				ptTip = pt.y * 0x10000;
				ptTip = ptTip + pt.x;

				// These two messages will set the position of the tooltip:
				WindowsAPI.SendMessage( m_lTTHwnd, (int)ToolTipMessages.TTM_TRACKPOSITION, 0, ptTip );
				WindowsAPI.SendMessage( m_lTTHwnd, (int)ToolTipMessages.TTM_TRACKACTIVATE, Convert.ToInt32(true), ref ti );
			}
		}

		public void CreateToolTip( 
			Control Parent, 
			ToolTipStyle style,
			bool popUpOnDemand
			)
		{
			int lWinStyle = 0;
			
			if ( m_lTTHwnd != IntPtr.Zero )
			{
				WindowsAPI.DestroyWindow( m_lTTHwnd );
			}

			m_lParentHwnd = Parent.Handle;

			// constant properties
			m_Style         = style;
			m_PopupOnDemand = popUpOnDemand;

			// create baloon style if desired
			if ( m_Style == ToolTipStyle.TT_BALLOON )
				lWinStyle = lWinStyle | (int)ToolTipStyleFlags.TTS_BALLOON;

			int lpParam=0;
			m_lTTHwnd = WindowsAPI.CreateWindowEx( 0, 
				TOOLTIPS_CLASSA, 
				string.Empty, lWinStyle, 
				0, 0, 0, 0,
				m_lParentHwnd.ToInt32(), 0, 0, ref lpParam );

			// now set our tooltip info structure
			// NOTE: dont incude TTF_SUBCLASS for on demand
			// if we want it centered, then set that flag
			if ( m_Centered )
			{
				if ( m_PopupOnDemand == false )
					ti.lFlags = (int)ToolTipFlags.TTF_SUBCLASS | (int)ToolTipFlags.TTF_CENTERTIP | (int)ToolTipFlags.TTF_IDISHWND;
				else
					ti.lFlags = (int)ToolTipFlags.TTF_IDISHWND | (int)ToolTipFlags.TTF_TRACK | (int)ToolTipFlags.TTF_CENTERTIP;
			}
			else
			{
				if ( m_PopupOnDemand == false )
					ti.lFlags = (int)ToolTipFlags.TTF_SUBCLASS | (int)ToolTipFlags.TTF_IDISHWND;
				else
					ti.lFlags = (int)ToolTipFlags.TTF_IDISHWND | (int)ToolTipFlags.TTF_TRACK | (int)ToolTipFlags.TTF_TRANSPARENT;
			}

			// set the hwnd prop to our parent control's hwnd
			ti.hwnd = m_lParentHwnd;
			ti.lId  = m_lParentHwnd.ToInt32(); // 0
			ti.hInstance = 0;        // VB6.GetHInstance.ToInt32
			ti.lpStr = m_TipText;
			ti.lSize = System.Runtime.InteropServices.Marshal.SizeOf( ti );
			
			// add the tooltip structure
			WindowsAPI.SendMessage( m_lTTHwnd, (int)ToolTipMessages.TTM_ADDTOOLA, 0, ref ti );

			// set all other property of tooltip
			Title = m_Title;
			if ( m_BackColor   != 0 ) BackColor     = m_BackColor;
			if ( m_ForeColor   != 0 ) ForeColor     = m_ForeColor;
			if ( m_MaxTipWidth != 0 ) MaxTipWidth   = m_MaxTipWidth;
			if ( m_VisibleTime != 0 ) VisibleTime   = m_VisibleTime;
			if ( m_DelayTime   != 0 ) DelayTime     = m_DelayTime;
			if ( m_TipFont     != null ) Font       = m_TipFont;
		}
		#endregion

		#region IDisposable Members

		public void Dispose()
		{
			if ( m_lTTHwnd != IntPtr.Zero )
			{
				WindowsAPI.DestroyWindow( m_lTTHwnd );
				m_lTTHwnd = IntPtr.Zero;
			}
		}

		#endregion
	}
}
