﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.Xml;

namespace FileEncryptionWithPasswordRecovery
{
    public delegate string AskForFilenameToSave();
    public delegate string AskForNewPasswordDelegate( out string Email );
    public delegate string AskForExistingPasswordDelegate( string FileName, string Email, string EmailPasswordHash, out string UnlockCode );

    public class ApplicationDocument
    {
        public ApplicationDocument() 
        {
            FileName     = string.Empty;
            DocumentData = string.Empty;
        }

        public ApplicationDocument( string FileName, AskForExistingPasswordDelegate AskForPassword )
        {
            if ( !File.Exists( FileName ) ) throw new ApplicationException( "Failed to load the document" );

            /* there are two possibilities
             * 
             * the filedata is encrypted or not
             */
            this.FileName = FileName;

            FileStream fs = File.Open( FileName, FileMode.Open, FileAccess.Read );
            try
            {
                /* if data is xml and contain known nodes - it's encrypted */
                XmlDocument encryptedFs = isDocumentDataEncrypted( fs );
                if ( encryptedFs == null )
                {
                    /* plain data */
                    StreamReader sr = new StreamReader( fs, Encoding.UTF8 );

                    UserEmail = string.Empty;
                    DocumentData = sr.ReadToEnd();
                }
                else
                {
                    /* read user email from email section */
                    string EmailPasswordHash = encryptedFs.SelectSingleNode( "file/signature" ).FirstChild.Value;
                    UserEmail = Encoding.UTF8.GetString( Convert.FromBase64String( encryptedFs.SelectSingleNode( "file/email" ).FirstChild.Value ) );

                    /* present the email and ask for password */
                    string unlockCode;
                    Password = AskForPassword( FileName, UserEmail, EmailPasswordHash, out unlockCode );
                    if ( string.IsNullOrEmpty( Password ) ) throw new ApplicationException( "Password must be provided!" );

                    /* got unlock code? */
                    if ( !string.IsNullOrEmpty( unlockCode ) )
                        /* override the actual password */
                        this.ActualPassword = unlockCode;

                    /* compute actual password to decode file content */
                    EncryptionHelper encryptor = new EncryptionHelper();

                    string EncryptedDocumentData = encryptedFs.SelectSingleNode( "file/content" ).FirstChild.Value;

                    /* decode file data */
                    DocumentData = encryptor.DESDecrypt( EncryptedDocumentData, ActualPassword );

                    this.IsEncrypted = true;
                }
            }
            finally
            {
                if ( fs != null ) fs.Close();
            }
        }

        public string FileName     { get; private set; }
        public string DocumentData { get; set; }
        public string UserEmail    { get; set; }
        public bool   IsEncrypted    { get; set; }

        public const string DUMMYPASSWORD = "thisisadummypasswordsetwhenrecoveringwithunlockcode";

        private string password;
        public string Password
        {
            get { return password; }
            set
            {
                password = value;

                if ( value != null )
                {
                    /* save encrypted content */
                    EncryptionHelper encryptor = new EncryptionHelper();
                    /* the actual password is SHA512( userpassword ) */
                    ActualPassword = encryptor.ComputePasswordHash( value );
                }
            }
        }
        public string ActualPassword { get; set; }

        /// <summary>
        /// Document data is encrypted if it's an XML and contains known nodes
        /// </summary>
        /// <param name="DocumentData"></param>
        /// <returns></returns>
        private XmlDocument isDocumentDataEncrypted( Stream DocumentData )
        {
            try
            {
                XmlDocument doc = new XmlDocument();

                doc.Load( DocumentData );

                if ( doc.SelectSingleNode( "file/signature" ) != null &&
                     doc.SelectSingleNode( "file/email" ) != null &&
                     doc.SelectSingleNode( "file/content" ) != null 
                    )
                    return doc;

                return null;
            }
            catch
            {
                return null;
            }
            finally
            {
                DocumentData.Seek( 0, SeekOrigin.Begin );
            }
        }

        /// <summary>
        /// Save document data
        /// </summary>
        /// <param name="AskForFileName"></param>
        public void Save( AskForFilenameToSave AskForFileName )
        {
            /* ask for filename if not set */
            if ( string.IsNullOrEmpty( this.FileName ) )
            {
                FileName = AskForFileName();

                /* abort if no filename */
                if ( string.IsNullOrEmpty( FileName ) ) 
                    return;
            }

            /* save document content */
            if ( !IsEncrypted )
            {
                /* not encrypted - save as plain data */
                StreamWriter sr = new StreamWriter( FileName, false, Encoding.UTF8 );
                sr.Write( DocumentData );

                sr.Close();
            }
            else
            {
                /* encrypted - save as encrypted content,
                 * password and email are known so pass them 
                 */
                SaveWithPassword( 
                    AskForFileName, 
                    ( out string email ) => 
                    { 
                        email = this.UserEmail; return this.Password; 
                    } 
                );
            }
        }

        /// <summary>
        /// Save encrypted document data
        /// </summary>
        /// <param name="AskForFileName"></param>
        /// <param name="AskForPassword"></param>
        public bool SaveWithPassword( AskForFilenameToSave AskForFileName, AskForNewPasswordDelegate AskForPassword )
        {
            /* ask for filename if not set */
            if ( string.IsNullOrEmpty( this.FileName ) )
            {
                FileName = AskForFileName();

                /* abort if no filename */
                if ( string.IsNullOrEmpty( FileName ) )
                    return false;
            }

            /* ask for password if not set */
            if ( string.IsNullOrEmpty( this.Password ) )
            {
                string email;
                
                this.Password  = AskForPassword( out email );
                this.UserEmail = email;

                /* abort if no filename */
                if ( string.IsNullOrEmpty( Password ) )
                    return false;
            }

            EncryptionHelper encryptor = new EncryptionHelper();

            FileStream    fs = new FileStream( FileName, FileMode.Create, FileAccess.Write );
            XmlTextWriter xw = new XmlTextWriter( fs, Encoding.UTF8 );
            xw.WriteStartDocument();

            xw.WriteStartElement( "file" );

            /* email is stored in the email section */
            xw.WriteStartElement( "email" );
            xw.WriteValue( Convert.ToBase64String( Encoding.UTF8.GetBytes( this.UserEmail ) ) );
            xw.WriteEndElement();

            /* email & password are stored in signature section, encrypted with RSA */
            string signature = string.Format( "{0}-|-{1}", this.UserEmail, ActualPassword );
            xw.WriteStartElement( "signature" );
            xw.WriteValue( encryptor.RSAEncrypt( signature ) );
            xw.WriteEndElement();

            /* encrypted content is stored in content section, encrypted with DES */
            xw.WriteStartElement( "content" );
            xw.WriteValue( encryptor.DESEncrypt( this.DocumentData, ActualPassword ) );
            xw.WriteEndElement();

            /* ending */
            xw.WriteEndElement();

            xw.WriteEndDocument();

            xw.Flush();
            xw.Close();

            fs.Close();
            fs.Dispose();

            return true;
        }
    }
}
