﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Microsoft.Xna.Framework.Graphics;
using XNADash.Textures;
using XNADash.Sound;

namespace XNADash.BoardBlocks
{
    public class BaseBlock
    {
        public int X;
        public int Y;
        public bool Moved = false;

        public DashBoard Board { get; set; }

        public void Open()
        {
            this._canBeConsumed = true;
        }

        private bool _canBeConsumed = false;
        public virtual bool CanBeConsumed
        {
            get
            {
                return _canBeConsumed;
            }
        }

        public virtual bool DoesFall
        {
            get
            {
                return false;
            }
        }

        public virtual bool CanBePushed
        {
            get
            {
                return false;
            }
        }

        public virtual bool CanExplode
        {
            get
            {
                return true;
            }
        }

        public virtual bool TriggersExplosion
        {
            get
            {
                return true;
            }
        }

        public virtual bool OthersFallFrom
        {
            get
            {
                return false;
            }
        }

        protected virtual GameTexture BlockTexture
        {
            get
            {
                return GameTexture.Empty;
            }
        }

        public virtual Texture2D Texture
        {
            get
            {
                return TextureFactory.Instance.GetTexture( this.BlockTexture );
            }
        }

        public BaseBlock GetNeighbour( Directions Direction )
        {
            return GetNeighbour( GetNeighbourDeltaX( Direction ), GetNeighbourDeltaY( Direction ) );
        }

        public BaseBlock GetNeighbour( int dX, int dY )
        {
            int NewX = this.X + dX;
            int NewY = this.Y + dY;

            if ( NewX < 0 || NewX > DashBoard.BOARDSIZEX - 1 ||
                 NewY < 0 || NewY > DashBoard.BOARDSIZEY - 1
                )
                return new OutOfBoardBlock();

            return Board.Blocks
                .Where(
                    b =>
                        b.X == NewX &&
                        b.Y == NewY
                    )
                .FirstOrDefault();
        }

        public void MoveTo( Directions Direction )
        {
            this.X += GetNeighbourDeltaX( Direction );
            this.Y += GetNeighbourDeltaY( Direction );
            this.Moved = true;
        }

        protected int GetNeighbourDeltaX( Directions Direction )
        {
            switch ( Direction )
            {
                case Directions.NW:
                case Directions.W:
                case Directions.SW:
                    return -1;

                case Directions.NE:
                case Directions.E:
                case Directions.SE:
                    return +1;

                default:
                    return 0;
            }
        }

        protected int GetNeighbourDeltaY( Directions Direction )
        {
            switch ( Direction )
            {
                case Directions.NW:
                case Directions.N:
                case Directions.NE: 
                    return -1; 

                case Directions.SW: 
                case Directions.S: 
                case Directions.SE:
                    return +1;

                default:
                    return 0;
            }
        }

        protected bool IsFalling = false;
        public virtual bool ApplyPhysics()
        {
            // czy pod spodem coś jest?
            BaseBlock block = this.GetNeighbour( Directions.S );
            if ( block == null )
            {
                // bezwarunkowe spadanie
                this.MoveTo( Directions.S );
                IsFalling = true;

                return false;
            }
            else
            {
                // spadł na gracza
                if ( block is PlayerBlock && this.IsFalling )
                {
                    block.ExplodeNeighbour( Directions.None );
                    SoundFactory.Instance.PlayEffect( SoundType.Stone );
                }

                // stoi na kamieniu
                if (
                     ( block.DoesFall && !block.IsFalling ) ||
                     ( block.OthersFallFrom ) 
                    )
                {
                    // może spaść w prawo - lewo?
                    if ( this.GetNeighbour( Directions.E ) == null &&
                         this.GetNeighbour( Directions.SE ) == null &&
                         ( this.GetNeighbour( Directions.NE ) == null || !this.GetNeighbour( Directions.NE ).DoesFall )
                        )
                    {
                        this.MoveTo( Directions.E );
                        IsFalling = true;

                        return false;
                    }
                    if ( this.GetNeighbour( Directions.W ) == null &&
                         this.GetNeighbour( Directions.SW ) == null &&
                         ( this.GetNeighbour( Directions.NW ) == null || !this.GetNeighbour( Directions.NW ).DoesFall )
                        )
                    {
                        this.MoveTo( Directions.W );
                        IsFalling = true;

                        return false;
                    }
                }
            }

            IsFalling = false;
            return true;
        }

        public void ExplodeNeighbour( Directions Direction )
        {
            BaseBlock neighbour = this.GetNeighbour( Direction );
            if ( neighbour != null &&
                 neighbour.CanExplode
                )
            {
                if (  neighbour is BombBlock &&
                    ( Direction == Directions.W || Direction == Directions.E || Direction == Directions.S || Direction == Directions.N )
                    )
                {
                    BombBlock bombNeighbour = neighbour as BombBlock;
                    bombNeighbour.MustExplode = true;
                }
                else 
                {
                    this.Board.RemoveBlock( neighbour );
                    this.Board.AddBlock( new BoomBlock() { X = neighbour.X, Y = neighbour.Y } );
                }
            }
            if ( neighbour == null )
            {
                this.Board.AddBlock( new BoomBlock() { X = this.X + this.GetNeighbourDeltaX( Direction ), Y = this.Y + this.GetNeighbourDeltaY( Direction ) } );
            }

        }
    }

    public enum Directions { None, NW, N, NE, W, E, SW, S, SE }
}
