/*
 * Trilma board is represented as a 21x21 matrix:
 * .....................
 * .....................
 * ..............O......
 * .............OO......
 * ............OOO......
 * ...........OOOO......
 * ......OOOOOOOOOOOOO..
 * ......OOOOOOOOOOOO...
 * ......OOOOOOOOOOO....
 * ......OOOOOOOOOO.....
 * ......OOOOOOOOO......
 * .....OOOOOOOOOO......
 * ....OOOOOOOOOOO......
 * ...OOOOOOOOOOOO......
 * ..OOOOOOOOOOOOO......
 * ......OOOO...........
 * ......OOO............
 * ......OO.............
 * ......O..............
 * .....................
 * .....................
 * 
 * Each square has 6 neighbours: N, NW, W, S, SE, E
 * Players are numbered 
 */
using System;
using System.Collections;
using System.Diagnostics; 
using System.Drawing;
using System.Drawing.Drawing2D;
using System.IO;
using System.Threading;
using System.Windows.Forms;

namespace NTRILMA
{	
	public class TrilmaBoard
	{
		// consts
		// consts
		public const int NO_PLAYERS     = 6;
		public const int NO_PLAYERS_MIN = 2;
		public const int NO_PLAYER_PCS  = 10;
		
		public const int BOARD_SIZE   = 21;
		const int BOARD_MARGIN = 2;
		const int BOARD_TR_LEN = 13; // board triangle
		const int BOARD_SQ_LEN = 9;  // board big center square
		const int BOARD_TR_OFS = 6;
		const int BOARD_CHK_LR = 4;  // checkers' layers

		const int BOARD_PL_TR  = 4;  // player pieces triangle
		
		const int BITMAP_EP  = 22;    // endpoint circle
		const int BITMAP_CS  = 15;    // piece size
		const int BITMAP_CSE = 10;    // empty square
		
		const int SQUARE_N   = 6;     // square neighbours
		
		const int TH_SLEEP   = 300;   // sleep time
		
		// variables
		public TrilmaSquare[,] theBoard;
		public TrilmaPlayer[]  trilmaPlayers;
		public bool            trilmaMega;
		public TrilmaMoveSequence humanMove = new TrilmaMoveSequence();
		
		public int             currentPlayer=0;
		public ArrayList       finishedPlayers=new ArrayList();
		
		// delegates
		public delegate void TrilmaBoardRefresh();

		#region Construction		
		// constructors
		public TrilmaBoard()
		{
			int i, j;

			// initialize the array of players
			trilmaPlayers = new TrilmaPlayer[ NO_PLAYERS ];
			for ( i=0; i<NO_PLAYERS; i++ )
				trilmaPlayers[i] = new TrilmaPlayer(i);

			theBoard = new TrilmaSquare[ BOARD_SIZE, BOARD_SIZE ];
			for ( i=0; i<BOARD_SIZE; i++ )
				for ( j=0; j<BOARD_SIZE; j++ )
					theBoard[i,j] = new TrilmaSquare();
			
			InitializeBoard();
		}
		
		public void InitializeBoard()
		{
			int i, j;
			// currentPlayer
			currentPlayer = 0;
			// first: set all squares to OutsideBoard
			for ( i=0; i<BOARD_SIZE; i++ )
				for ( j=0; j<BOARD_SIZE; j++ )
				{
					theBoard[i,j].Status = TrilmaSquareStatus.OutsideBoard;
					theBoard[i,j].Piece  = TrilmaSquare.EMPTY;
				}

			// second: mark two triangles as InsideBoard
			for ( i=0; i<BOARD_TR_LEN; i++ )
				for ( j=0; j<BOARD_TR_LEN; j++ )
					if ( i+j < BOARD_TR_LEN )
					{
						theBoard[i+BOARD_TR_OFS, j+BOARD_TR_OFS].Status =
							TrilmaSquareStatus.InsideBoard;
						theBoard[BOARD_SIZE-(i+BOARD_TR_OFS+1), BOARD_SIZE-(j+BOARD_TR_OFS+1)].Status =
							TrilmaSquareStatus.InsideBoard;
					}			
			
			// start position
			for ( i=0; i<BOARD_PL_TR; i++ )
				for ( j=0; j<BOARD_PL_TR; j++ )
					if ( i+j < BOARD_PL_TR )
					{
						// player 0
						trilmaPlayers[0].plStartPos.Add( new Point( i+BOARD_TR_OFS, j+BOARD_TR_OFS ) );
						// player 1						
						trilmaPlayers[1].plStartPos.Add( new Point( BOARD_SIZE-(i+BOARD_TR_OFS+1), BOARD_SIZE-(j+BOARD_TR_OFS+BOARD_SQ_LEN+1) ) );
						// player 2
						trilmaPlayers[2].plStartPos.Add( new Point( i+BOARD_TR_OFS+BOARD_SQ_LEN, j+BOARD_TR_OFS ) );
						// player 3
						trilmaPlayers[3].plStartPos.Add( new Point( BOARD_SIZE-(i+BOARD_TR_OFS+1), BOARD_SIZE-(j+BOARD_TR_OFS+1) ) );
						// player 4
						trilmaPlayers[4].plStartPos.Add( new Point( i+BOARD_TR_OFS, j+BOARD_TR_OFS+BOARD_SQ_LEN ) );
						// player 5						
						trilmaPlayers[5].plStartPos.Add( new Point( BOARD_SIZE-(i+BOARD_TR_OFS+BOARD_SQ_LEN+1), BOARD_SIZE-(j+BOARD_TR_OFS+1) ) );
					}
					
			if ( this.trilmaMega )
				for ( i=0; i<BOARD_PL_TR-1; i++ )
					for ( j=0; j<BOARD_PL_TR-1; j++ )
					{
						if ( i+j == BOARD_PL_TR-2 )
						{
							// player 0
							trilmaPlayers[0].plStartPos.Add( new Point( BOARD_TR_OFS+BOARD_CHK_LR-i-1, BOARD_TR_OFS+BOARD_CHK_LR-j-1 ) );
							// player 3
							trilmaPlayers[3].plStartPos.Add( new Point( i+BOARD_TR_OFS+BOARD_CHK_LR+1, j+BOARD_TR_OFS+BOARD_CHK_LR+1 ) );
						}
						if ( j == 0 )
						{
							// player 1						
							trilmaPlayers[1].plStartPos.Add( new Point( i+BOARD_TR_OFS+BOARD_CHK_LR+1, j+BOARD_TR_OFS ) );
							// player 4
							trilmaPlayers[4].plStartPos.Add( new Point( BOARD_TR_OFS+BOARD_CHK_LR-i-1, BOARD_TR_OFS+2*BOARD_CHK_LR-j ) );							
						}
						if ( i == 0 )
						{
							// player 2
							trilmaPlayers[2].plStartPos.Add( new Point( BOARD_TR_OFS+2*BOARD_CHK_LR-i, BOARD_TR_OFS+BOARD_CHK_LR-j-1 ) );
							// player 5						
							trilmaPlayers[5].plStartPos.Add( new Point( i+BOARD_TR_OFS, j+BOARD_TR_OFS+BOARD_CHK_LR+1 ) );							
						}
					}
					
            // end position
            trilmaPlayers[0].plEndPos = trilmaPlayers[3].plStartPos;
            trilmaPlayers[1].plEndPos = trilmaPlayers[4].plStartPos;
            trilmaPlayers[2].plEndPos = trilmaPlayers[5].plStartPos;
            trilmaPlayers[3].plEndPos = trilmaPlayers[0].plStartPos;
            trilmaPlayers[4].plEndPos = trilmaPlayers[1].plStartPos;
            trilmaPlayers[5].plEndPos = trilmaPlayers[2].plStartPos;

			// inner, outer points
			trilmaPlayers[0].plStartPoint = new Point( BOARD_TR_OFS, BOARD_TR_OFS );
			trilmaPlayers[1].plStartPoint = new Point( BOARD_SIZE-BOARD_TR_OFS-1, BOARD_SIZE-BOARD_TR_OFS-BOARD_TR_LEN );
			trilmaPlayers[2].plStartPoint = new Point( BOARD_TR_OFS+BOARD_TR_LEN-1, BOARD_TR_OFS );
			trilmaPlayers[3].plStartPoint = new Point( BOARD_SIZE-BOARD_TR_OFS-1, BOARD_SIZE-BOARD_TR_OFS-1 );
			trilmaPlayers[4].plStartPoint = new Point( BOARD_TR_OFS, BOARD_TR_OFS+BOARD_TR_LEN-1 );
			trilmaPlayers[5].plStartPoint = new Point( BOARD_SIZE-BOARD_TR_OFS-BOARD_TR_LEN, BOARD_SIZE-BOARD_TR_OFS-1 );

			trilmaPlayers[0].plEndPoint = trilmaPlayers[3].plStartPoint;
			trilmaPlayers[1].plEndPoint = trilmaPlayers[4].plStartPoint;
			trilmaPlayers[2].plEndPoint = trilmaPlayers[5].plStartPoint;
			trilmaPlayers[3].plEndPoint = trilmaPlayers[0].plStartPoint;
			trilmaPlayers[4].plEndPoint = trilmaPlayers[1].plStartPoint;
			trilmaPlayers[5].plEndPoint = trilmaPlayers[2].plStartPoint;

			// colors		
			trilmaPlayers[0].plColor = Color.Red;
			trilmaPlayers[1].plColor = Color.Blue;
			trilmaPlayers[2].plColor = Color.Gold;
			trilmaPlayers[3].plColor = Color.Green;
			trilmaPlayers[4].plColor = Color.Chocolate;
			trilmaPlayers[5].plColor = Color.DarkViolet; 
						
			// setup board
			for ( i=0; i<TrilmaBoard.NO_PLAYERS; i++ )
				if ( trilmaPlayers[i].plStatus != TrilmaPlayerStatus.None )
					foreach ( Point startPoint in trilmaPlayers[i].plStartPos )
						theBoard[ startPoint.X, startPoint.Y ].Piece = i;		
			
			finishedPlayers.Clear();			
		}
		#endregion
		#region Board management

		/// <summary>
		/// Calculate the distance between two given points on the board
		/// If the 
		/// </summary>
		public int CalculateDistance( Point sourceP, Point destP )
		{
			int dx = destP.X - sourceP.X;
			int dy = destP.Y - sourceP.Y;
						
			if ( dx*dy >= 0 ) 
				return Math.Abs( dx ) + Math.Abs( dy );
			else 
				return Math.Max( Math.Abs( dx ), Math.Abs( dy ) );
		}

		/// <summary>
		/// Returns an ArrayList of Point object that holds
		/// all given point's EMPTY far neighbours
		/// </summary>
		public ArrayList GetFarNeighbours( Point squarePos )
		{
			int i, j;
			
	   		ArrayList aNeigh = new ArrayList();			
			// test all neighbours
			if ( theBoard[ squarePos.X, squarePos.Y ].Status ==
			     TrilmaSquareStatus.InsideBoard )
				for ( i=-1; i<=1; i++ )
					for ( j=-1; j<=1; j++ )
					{
						if ( (i==0  && j==0 ) ||
						     (i==1 && j==1 ) ||
						     (i==-1  && j==-1) )
						     continue;
						
						Point neighFarPos = 
							new Point( squarePos.X+2*i, squarePos.Y+2*j );
						Point neighPos = 
							new Point( squarePos.X+i, squarePos.Y+j );
						
						if ( 
						     theBoard[ neighFarPos.X, neighFarPos.Y ].Status != TrilmaSquareStatus.OutsideBoard &&
						     theBoard[ neighPos.X, neighPos.Y ].Status != TrilmaSquareStatus.OutsideBoard 
						    )
							if ( 
							     theBoard[ neighFarPos.X, neighFarPos.Y ].Piece == TrilmaSquare.EMPTY &&
							     theBoard[ neighPos.X, neighPos.Y ].Piece != TrilmaSquare.EMPTY
                                )  
								aNeigh.Add( neighFarPos );
					}
				
			return aNeigh;
		}
		/// <summary>
		/// Returns an ArrayList of Point object that holds
		/// all given point's neighbours
		/// </summary>
		public ArrayList GetNearNeighbours( Point squarePos, bool bEmptyOnly )
		{
			int i, j;
			
	   		ArrayList aNeigh = new ArrayList();			
			// test all neighbours
			if ( theBoard[ squarePos.X, squarePos.Y ].Status ==
			     TrilmaSquareStatus.InsideBoard )
				for ( i=-1; i<=1; i++ )
					for ( j=-1; j<=1; j++ )
					{
						if ( (i==0  && j==0 ) ||
						     (i==1 && j==1 ) ||
						     (i==-1  && j==-1) )
						     continue;
						
						Point neighPos = 
							new Point( squarePos.X+i, squarePos.Y+j );
						
						if ( theBoard[ neighPos.X, neighPos.Y ].Status != TrilmaSquareStatus.OutsideBoard )
							if ( bEmptyOnly == false ||
							     ( bEmptyOnly == true &&
						           theBoard[ neighPos.X, neighPos.Y ].Piece == TrilmaSquare.EMPTY 
                                 )						         
                                )  
								aNeigh.Add( neighPos );
					}
				
			return aNeigh;
		}
		public bool AreValidNearNeighbours( Point sPoint, Point dPoint )
		{
			ArrayList nPoints = GetNearNeighbours( sPoint, true );
			foreach ( Point np in nPoints )
				if ( np.X == dPoint.X && np.Y == dPoint.Y )
					return true;
			return false;
		}
		public bool AreValidFarNeighbours( Point sPoint, Point dPoint )
		{
			ArrayList nPoints = GetFarNeighbours( sPoint );
			foreach ( Point np in nPoints )
				if ( np.X == dPoint.X && np.Y == dPoint.Y )
					return true;
			return false;
		}
		#endregion
				
		#region Moves
		public bool IsPlayerMoveValid()
		{
			// I assume that human will never be blocked
			if ( humanMove.Moves.Count < 1 ) return false;

			// First Move
			TrilmaMove sMove = (TrilmaMove)humanMove.Moves[0];
			if ( theBoard[sMove.srcPoint.X, sMove.srcPoint.Y].Piece != currentPlayer ) return false;

			// single jump
			if ( humanMove.Moves.Count == 2 )
			{
				TrilmaMove pMove = (TrilmaMove)humanMove.Moves[1];
				return AreValidNearNeighbours( pMove.srcPoint, pMove.destPoint ) ||
				       AreValidFarNeighbours( pMove.srcPoint, pMove.destPoint );
			}
			else // sequence of jumps
			{
				for ( int k=1; k<humanMove.Moves.Count; k++ )
				{
					TrilmaMove pMove = (TrilmaMove)humanMove.Moves[k];
					if ( AreValidFarNeighbours( pMove.srcPoint, pMove.destPoint ) == false )
						return false;
				}
				return true;
			}
		}
		public TrilmaMoveSequence GetNextMove()
		{						
			ArrayList aAllMoves = GetAllMoves();

			Random r = new Random();			
			if ( aAllMoves.Count == 0 ) 
			{
				return null;
			}
			else
			{
				TrilmaMoveSequence tRetMove = (TrilmaMoveSequence)aAllMoves[0];
								
				// the less value the better the move is
				foreach ( TrilmaMoveSequence tMove in aAllMoves )
				{					
					int tRetMoveVal = tRetMove.MoveValue( this );
					int tMoveVal    = tMove.MoveValue( this );

					// if the move is better or equal, take it						
					if ( tMoveVal > tRetMoveVal ||
					     ( (tMoveVal == tRetMoveVal) && (r.Next()%2==0) ) )
						tRetMove = tMove;
				}								
				return tRetMove;
			}
		}
		///<summary>
		/// Returns an ArrayList of TrilmaMoveSequence
		/// containing all moves from given position
		/// </summary>
		ArrayList GetAllMoves()
		{
			ArrayList aMoves = new ArrayList();
			
			int i, j;
			
			for ( i=0; i<BOARD_SIZE; i++ )
				for ( j=0; j<BOARD_SIZE; j++ )
				{
					if ( theBoard[i,j].Piece == currentPlayer )
					{
						Point plPiece = new Point( i, j );
						
						// get all near neighbours
						ArrayList nN = GetNearNeighbours( plPiece, true );												
						foreach ( Point dPoint in nN )
						{
							TrilmaMoveSequence tMoveSeq = new TrilmaMoveSequence();								
							TrilmaMove tMoveStep = new TrilmaMove( plPiece, dPoint );								
							tMoveSeq.Moves.Add( tMoveStep );								
							aMoves.Add( tMoveSeq );
						}
						// get far neighbours and try to expand
						// them recursively
						ArrayList fN = GetFarNeighbours( plPiece );
						foreach ( Point dPoint in fN )
						{
							TrilmaMoveSequence tMoveSeq = new TrilmaMoveSequence();								
							TrilmaMove tMoveStep = new TrilmaMove( plPiece, dPoint );
							tMoveSeq.Moves.Add( tMoveStep );
							aMoves.Add( tMoveSeq );
							BuildMoveList( tMoveSeq, aMoves );																					
						}

					}
				}
				
			return aMoves;
		}
		#region Long move sequences
		void BuildMoveList( TrilmaMoveSequence tMoveSeq, ArrayList aMoveList )
		{						
			TrilmaMove tLastMoveInSeq = (TrilmaMove)tMoveSeq.Moves[tMoveSeq.Moves.Count-1];		
			
			foreach ( Point tNextSquare in GetFarNeighbours( tLastMoveInSeq.destPoint ) )
			{
				// does tMoveSeq contain tNextSquare?
				bool bAlreadyVisited = false;
				foreach ( TrilmaMove tCheckMove in tMoveSeq.Moves )
					if ( tNextSquare == tCheckMove.srcPoint )
						bAlreadyVisited = true;
				
				if ( bAlreadyVisited == false )
				{
					// if not - build new move
					TrilmaMoveSequence tNewMoveSeq = new TrilmaMoveSequence(tMoveSeq);
					TrilmaMove         tNewMove    = new TrilmaMove( tLastMoveInSeq.destPoint, tNextSquare );
					tNewMoveSeq.Moves.Add( tNewMove );
					// add it to the list
					aMoveList.Add( tNewMoveSeq );
					// call recursively
					BuildMoveList( tNewMoveSeq, aMoveList );
				}
			}
		}
		#endregion
		public void MakeMove( TrilmaMoveSequence trilmaMoveList, TrilmaBoardRefresh trilmaBoardRefresh )
		{
			// get next move
			if ( L_PlayerFinished( currentPlayer ) == false )
			{
				foreach ( TrilmaMove trilmaMove in trilmaMoveList.Moves )
				{										                
					theBoard[ trilmaMove.srcPoint.X, trilmaMove.srcPoint.Y ].Piece = TrilmaSquare.EMPTY;
					theBoard[ trilmaMove.destPoint.X, trilmaMove.destPoint.Y ].Piece = currentPlayer;
					// refresh the board
					Thread.Sleep( TH_SLEEP );				
					trilmaBoardRefresh();
				}
			}
			// final refresh of the board
			Thread.Sleep( TH_SLEEP );				
			trilmaBoardRefresh();
			// check if the player finished his game
			if ( L_PlayerFinished( currentPlayer ) &&
			     !finishedPlayers.Contains( currentPlayer ) )
			     finishedPlayers.Add( currentPlayer );
		}
		public void ChangePlayer()
		{
			currentPlayer++;
			if ( currentPlayer >= NO_PLAYERS )
				currentPlayer=0;
		}
		#endregion
		
		#region Graph routines
		public void FillBoardBitmap( Bitmap bBitmap, Bitmap bBackground )
		{
			int i,j;
			int BITMAP_X = bBitmap.Width;
			int BITMAP_Y = bBitmap.Height;

			if ( bBitmap == null ) return;
			
			Graphics      g = Graphics.FromImage( bBitmap );			
			g.SmoothingMode = SmoothingMode.HighQuality; 
			g.Clear( SystemColors.Window );
						
			// Background
			if ( bBackground != null )
				g.DrawImage( bBackground, g.VisibleClipBounds );
			
			// Who's gonna play?
			DrawPiece( g, currentPlayer, bBitmap.Width-BITMAP_CS, BITMAP_CS );
			// Finishing sequence
			for ( int fS=0; fS<finishedPlayers.Count; fS++ )
				DrawPiece( g, (int)finishedPlayers[fS], BITMAP_CS, (fS+1)*BITMAP_CS );
			// Neighbour connections
			for ( i=0; i<BOARD_SIZE; i++ )
				for ( j=0; j<BOARD_SIZE; j++ )
					if ( theBoard[i,j].Status == TrilmaSquareStatus.InsideBoard )
					{						
						// neighbour connections
						PointF sCoords = BoardPointToBitmapPoint( bBitmap, i, j );
						ArrayList aNeigh = GetNearNeighbours( new Point( i, j ), false );
						foreach ( Point n in aNeigh )
						{
							PointF nCoords = BoardPointToBitmapPoint( bBitmap, n.X, n.Y );
							g.DrawLine( Pens.Black, 
							           sCoords, nCoords );
						}
					}
					
			// Endpoints
			for ( int ip=0; ip<NO_PLAYERS; ip++ )
			{
				// point
				Point  eP      = trilmaPlayers[ip].plEndPoint;
				PointF sCoords = BoardPointToBitmapPoint( bBitmap, eP.X, eP.Y );
				RectangleF rP  = new RectangleF( sCoords.X - BITMAP_EP/2, sCoords.Y-BITMAP_EP/2, BITMAP_EP, BITMAP_EP );

				using( SolidBrush b = new SolidBrush( trilmaPlayers[ip].plColor ) )
				{
					g.FillEllipse( b, rP );
				}				
			}

			// Pieces
			for ( i=0; i<BOARD_SIZE; i++ )
				for ( j=0; j<BOARD_SIZE; j++ )
					if ( theBoard[i,j].Status == TrilmaSquareStatus.InsideBoard )
					{
						// point
						PointF sCoords = BoardPointToBitmapPoint( bBitmap, i, j );
																		
						if ( theBoard[i,j].Piece == TrilmaSquare.EMPTY )
						{
							RectangleF rE = new RectangleF( sCoords.X - BITMAP_CSE/2, sCoords.Y-BITMAP_CSE/2, BITMAP_CSE, BITMAP_CSE );
							g.FillEllipse( Brushes.Black, rE ); 
						}
						else
						{
							DrawPiece( g, theBoard[i,j].Piece, sCoords.X, sCoords.Y );
						}						
					}
					
			// Player move sequence
			if ( humanMove.Moves.Count > 0 )
			{
				using( Pen p = new Pen( Color.Black, 3 ) )
				{
					// select player piece
					Point sPoint  = ((TrilmaMove)humanMove.Moves[0]).srcPoint;
					PointF SPoint = BoardPointToBitmapPoint( bBitmap, sPoint.X, sPoint.Y );
					Rectangle sR = new Rectangle( (int)SPoint.X - BITMAP_CS/2, (int)SPoint.Y-BITMAP_CS/2, BITMAP_CS, BITMAP_CS );
					g.DrawRectangle( p, sR );
					// show the move
					for ( int k=1; k<humanMove.Moves.Count; k++ ) 
					{
						TrilmaMove hMove = (TrilmaMove)humanMove.Moves[k];
						
						Point sPointD  = hMove.destPoint;
						PointF SPointD = BoardPointToBitmapPoint( bBitmap, sPointD.X, sPointD.Y );
						Rectangle sRD = new Rectangle( (int)SPointD.X - BITMAP_CS/2, (int)SPointD.Y-BITMAP_CS/2, BITMAP_CS, BITMAP_CS );
						g.DrawRectangle( p, sRD );
						
						StringFormat sf  = new StringFormat();
						sf.Alignment     = System.Drawing.StringAlignment.Center;
						sf.LineAlignment = System.Drawing.StringAlignment.Center;
						
						using( System.Drawing.Font f = new System.Drawing.Font( "Tahoma", 6, FontStyle.Bold ) )
							g.DrawString( k.ToString(), f, Brushes.Azure, sRD, sf );
					}
				}
			}
								
			g.Dispose();			
		}
		void DrawPiece( Graphics g, int piece, float x, float y )
		{
			RectangleF rP = new RectangleF( x-BITMAP_CS/2, y-BITMAP_CS/2, BITMAP_CS, BITMAP_CS );
			RectangleF rS = new RectangleF( x-BITMAP_CS/3, y-BITMAP_CS/3, BITMAP_CS/3, BITMAP_CS/3 );
							
			using( SolidBrush br = new SolidBrush( trilmaPlayers[piece].plColor ) )
			{
				g.FillEllipse( br, rP ); 
				g.DrawEllipse( Pens.Black, rP );								
				g.FillEllipse( Brushes.White, rS );
			}			
		}
		/// <summary>
		/// Maps points coordinates from the board matrix
		/// to bitmap points
		/// </summary>
		public PointF BoardPointToBitmapPoint( Bitmap bBitmap, int x, int y )
		{
			float BX = bBitmap.Width;
			float BY = bBitmap.Height;
						
			float bitStepX = BX / BOARD_SIZE;
			float bitStepY = BY / BOARD_SIZE;

			// the x offset depends on the point's row (ie. y)
			float xOffset = BOARD_SIZE/2 - y;
			
			float px = (x - xOffset/2) * bitStepX;
			float py = y * bitStepY;			
			
			return new PointF( px, py );
		}
		/// <summary>
		/// Maps points coordinates from bitmap coords
		/// to array coords
		/// </summary>
		public Point BitmapPointToBoardPoint( PictureBox pBitmap, int px, int py )
		{
			float BX = pBitmap.Width;
			float BY = pBitmap.Height;
						
			float bitStepX = BX / BOARD_SIZE;
			float bitStepY = BY / BOARD_SIZE;

			// the x offset depends on the point's row (ie. y)			
			int y = Convert.ToInt32( py / bitStepY );			
			float xOffset = BOARD_SIZE/2.0f - y;
			// this is weird, don't know why bitStepX/4, though it works best
			int x = Convert.ToInt32( ((px-bitStepX/4) / bitStepX) + xOffset/2 ) ;
			
			return new Point( x, y );
		}
		#endregion
		#region Misc routines
		public bool L_PlayerFinished( int playerNo )
		{
			TrilmaPlayer tPlayer = trilmaPlayers[ playerNo ];

			if ( tPlayer.plStatus == TrilmaPlayerStatus.None ) return true;
						
			foreach ( Point pEndPoint in tPlayer.plEndPos )
				if ( theBoard[ pEndPoint.X, pEndPoint.Y ].Piece != playerNo )
					return false;
			
			return true;
		}	
		public bool L_GameFinished()
		{
			int i;
			for ( i=0; i<NO_PLAYERS; i++)
				if ( L_PlayerFinished( i )== false ) return false;
			
			return true;
		}
		public int L_CountPlayers()
		{
			int i, plCount=0;
			
			for (i=0; i<NO_PLAYERS; i++)
				if ( trilmaPlayers[i].plStatus != TrilmaPlayerStatus.None )
					plCount++;
			
			return plCount;
		}
		#endregion	
		#region Misc
		public override string ToString()
		{
			int i,j;
			
			string sRet = "Player " + currentPlayer.ToString() + " to play" + Environment.NewLine;
						
			for ( i=0; i<BOARD_SIZE; i++ )
			{
				for ( j=0; j<BOARD_SIZE; j++ )
					if ( theBoard[j,i].Piece == TrilmaSquare.EMPTY )
						sRet += ".";
					else
						sRet += theBoard[j,i].Piece.ToString();
				sRet += Environment.NewLine;
			}		
					
			return sRet;
		}
		#endregion
	}
}
