using System;
using System.Diagnostics;
using System.Runtime.InteropServices;


namespace vicMazeGen
{

    [StructLayout(LayoutKind.Sequential)]
    public struct Complex : IComparable, ICloneable {

		// Original      : Ben Houston at ben@exocortex.org
		// Modifications : Torq314

        //-----------------------------------------------------------------------------------
        //-----------------------------------------------------------------------------------

        public double Re;
        public double Im;

        //-----------------------------------------------------------------------------------
        //-----------------------------------------------------------------------------------

        public Complex( double real, double imaginary ) {
            this.Re    = (double) real;
            this.Im    = (double) imaginary;
        }

        public Complex( Complex c ) {
            this.Re    = c.Re;
            this.Im    = c.Im;
        }

        static public Complex    FromRealImaginary( double real, double imaginary ) {
            Complex c;
            c.Re = (double) real;
            c.Im = (double) imaginary;
            return c;
        }

        static public Complex    FromModulusArgument( double modulus, double argument ) {
            Complex c;
            c.Re    = (double)( modulus * System.Math.Cos( argument ) );
            c.Im    = (double)( modulus * System.Math.Sin( argument ) );
            return c;
        }
        
        //-----------------------------------------------------------------------------------
        //-----------------------------------------------------------------------------------

        object    ICloneable.Clone() {
            return    new Complex( this );
        }
        public Complex    Clone() {
            return    new Complex( this );
        }
        
        //-----------------------------------------------------------------------------------
        //-----------------------------------------------------------------------------------

        public double    GetModulus() {
            double    x    = this.Re;
            double    y    = this.Im;
            return    (double) Math.Sqrt( x*x + y*y );
        }

        public double    GetModulusSquared() {
            double    x    = this.Re;
            double    y    = this.Im;
            return    (double) x*x + y*y;
        }

        public double    GetArgument() {
            return (double) Math.Atan2( this.Im, this.Re );
        }

        //-----------------------------------------------------------------------------------

        public Complex GetConjugate() {
            return FromRealImaginary( this.Re, -this.Im );
        }

        //-----------------------------------------------------------------------------------
        
        public static explicit operator Complex ( double f ) {
            Complex c;
            c.Re    = (double) f;
            c.Im    = (double) 0;
            return c;
        }

        public static explicit operator double ( Complex c ) {
            return (double) c.Re;
        }
        
        //-----------------------------------------------------------------------------------
        //-----------------------------------------------------------------------------------

        public static bool    operator==( Complex a, Complex b ) {
            return    ( a.Re == b.Re ) && ( a.Im == b.Im );
        }

        public static bool    operator!=( Complex a, Complex b ) {
            return    ( a.Re != b.Re ) || ( a.Im != b.Im );
        }

        /// <summary>
        /// Get the hash code of the complex number
        /// </summary>
        /// <returns></returns>
        public override int        GetHashCode() {
            return    ( this.Re.GetHashCode() ^ this.Im.GetHashCode() );
        }

        public override bool    Equals( object o ) {
            if( o is Complex ) {
                Complex c = (Complex) o;
                return   ( this == c );
            }
            return    false;
        }

        //-----------------------------------------------------------------------------------
        //-----------------------------------------------------------------------------------

        public int    CompareTo( object o ) {
            if( o is Complex ) {
                return    this.GetModulus().CompareTo( ((Complex)o).GetModulus() );
            }
            if( o is double ) {
                return    this.GetModulus().CompareTo( (double)o );
            }
            if( o is float ) {
                return    this.GetModulus().CompareTo( (float)o );
            }
            return    0;
        }

        //-----------------------------------------------------------------------------------
        //-----------------------------------------------------------------------------------

        public static Complex operator+( Complex a ) {
            return a;
        }

        public static Complex operator-( Complex a ) {
            a.Re    = -a.Re;
            a.Im    = -a.Im;
            return a;
        }

        public static Complex operator+( Complex a, double f ) {
            a.Re    = (double)( a.Re + f );
            return a;
        }

        public static Complex operator+( double f, Complex a ) {
            a.Re    = (double)( a.Re + f );
            return a;
        }

        public static Complex operator+( Complex a, Complex b ) {
            a.Re    = a.Re + b.Re;
            a.Im    = a.Im + b.Im;
            return a;
        }

        public static Complex operator-( Complex a, double f ) {
            a.Re    = (double)( a.Re - f );
            return a;
        }

        public static Complex operator-( double f, Complex a ) {
            a.Re    = (double)( a.Re - f );
            return a;
        }

        public static Complex operator-( Complex a, Complex b ) {
            a.Re    = a.Re - b.Re;
            a.Im    = a.Im - b.Im;
            return a;
        }

        public static Complex operator*( Complex a, double f ) {
            a.Re    = (double)( a.Re * f );
            a.Im    = (double)( a.Im * f );
            return a;
        }
        
        public static Complex operator*( double f, Complex a ) {
            a.Re    = (double)( a.Re * f );
            a.Im    = (double)( a.Im * f );
            return a;
        }
        
        public static Complex operator*( Complex a, Complex b ) {
            // (x + yi)(u + vi) = (xu  yv) + (xv + yu)i. 
            double    x = a.Re, y = a.Im;
            double    u = b.Re, v = b.Im;
            a.Re    = (double)( x*u - y*v );
            a.Im    = (double)( x*v + y*u );
            return a;
        }

        public static Complex operator/( Complex a, double f ) {
            if( f == 0 ) {
                throw new DivideByZeroException();
            }
            a.Re    = (double)( a.Re / f );
            a.Im    = (double)( a.Im / f );
            return a;
        }
        
        public static Complex operator/( Complex a, Complex b ) {
            double    x = a.Re,    y = a.Im;
            double    u = b.Re,    v = b.Im;
            double    denom = u*u + v*v;

            if( denom == 0 ) {
                throw new DivideByZeroException();
            }
            a.Re    = (double)( ( x*u + y*v ) / denom );
            a.Im    = (double)( ( y*u - x*v ) / denom );
            return a;
        }

        /// <summary>
        /// Parse a complex representation in this fashion: "( %f, %f )"
        /// </summary>
        /// <param name="s"></param>
        /// <returns></returns>
        static public Complex Parse( string s ) 
		{
            throw new NotImplementedException( "Complex Complex.Parse( string s ) is not implemented." );
        }
        
        public override string ToString() 
		{
            return    String.Format( "( {0}, {1}i )", this.Re, this.Im );
        }

        //-----------------------------------------------------------------------------------
        //-----------------------------------------------------------------------------------

        static public bool IsEqual( Complex a, Complex b, double tolerance ) {
            return
                ( Math.Abs( a.Re - b.Re ) < tolerance ) &&
                ( Math.Abs( a.Im - b.Im ) < tolerance );

        }
        
        //----------------------------------------------------------------------------------
        //----------------------------------------------------------------------------------

        static public Complex    Zero {
            get    {    return    new Complex( 0, 0 );    }
        }

        static public Complex    I {
            get {    return    new Complex( 0, 1 );    }
        }

        static public Complex    MaxValue {
            get {    return    new Complex( double.MaxValue, double.MaxValue );    }
        }

        static public Complex    MinValue {
            get {    return    new Complex( double.MinValue, double.MinValue );    }
        }


        //----------------------------------------------------------------------------------
        //----------------------------------------------------------------------------------
    }

}

