using System;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Windows.Forms;

namespace LogicImage
{
  // -----------------------------------------------------------
  // Main Form
  // -----------------------------------------------------------
  public class CMainForm : Form
  {
    CGame    game;
    MainMenu mainMenu;

    string   Title = "LogicImage.NET";

    // -----------------------------------------------------------
    // constructor
    // -----------------------------------------------------------
    public CMainForm()
    {
  		SetStyle(ControlStyles.UserPaint, true);  
			SetStyle(ControlStyles.AllPaintingInWmPaint, true); 
			SetStyle(ControlStyles.DoubleBuffer, true); 
			
      
      this.Size            = new Size( 350, 200 );
      this.FormBorderStyle = FormBorderStyle.FixedDialog;
      this.MaximizeBox     = this.MinimizeBox = false;
      this.Text            = Title;

      // events
      this.MouseDown      += new MouseEventHandler( this_MouseDown );
      this.MouseMove      += new MouseEventHandler( this_MouseMove );

      // init the menu
      mainMenu = new MainMenu();
      
      // file
      MenuItem mFile    = new MenuItem( "File" );

      MenuItem mNewGame = new MenuItem( "New game" );
      mNewGame.Click   += new EventHandler( menu_NewGame );
      mFile.MenuItems.Add( mNewGame );

      mFile.MenuItems.Add( new MenuItem("-") );

      MenuItem mLoadGame = new MenuItem( "Load game" );
      mLoadGame.Click   += new EventHandler( menu_LoadGame );
      mFile.MenuItems.Add( mLoadGame );

      MenuItem mSaveGame = new MenuItem( "Save game" );
      mSaveGame.Click   += new EventHandler( menu_SaveGame );
      mFile.MenuItems.Add( mSaveGame );

      mFile.MenuItems.Add( new MenuItem("-") );

      MenuItem mExit = new MenuItem( "Exit" );
      mExit.Click   += new EventHandler( menu_Exit );
      mFile.MenuItems.Add( mExit );

      mainMenu.MenuItems.Add( mFile );

      // game
      MenuItem mGame = new MenuItem( "Game" );

      MenuItem mUndo     = new MenuItem( "Undo" );
      mUndo.Shortcut     = Shortcut.CtrlZ;
			mUndo.ShowShortcut = true;
      mUndo.Click       += new EventHandler( menu_Undo );
      mGame.MenuItems.Add( mUndo );

      MenuItem mRedo     = new MenuItem( "Redo" );
      mRedo.Shortcut     = Shortcut.CtrlY;
			mRedo.ShowShortcut = true;
      mRedo.Click       += new EventHandler( menu_Redo );
      mGame.MenuItems.Add( mRedo );

      mainMenu.MenuItems.Add( mGame );

      // about
      MenuItem mAbout    = new MenuItem( "About..." );
      mAbout.Click      += new EventHandler( menu_About );
      mainMenu.MenuItems.Add( mAbout );

      this.Menu           = mainMenu;
    }

    // -----------------------------------------------------------
    // menu
    // -----------------------------------------------------------
    void menu_NewGame( object sender, EventArgs e )
    {
      OpenFileDialog of   = new OpenFileDialog();
			of.InitialDirectory = Application.StartupPath;
			of.CheckFileExists  = true;
      of.Filter           = "LogicImage board data (*.logic_image)|*.logic_image";

      DialogResult res = of.ShowDialog();
      if ( res == DialogResult.OK )
      {
        game = CGameLoader.NewGame( of.FileName );
        if ( game != null )
        {
          game.PrepareOwnerForm( this );     
        }
      }
    }

    void menu_LoadGame( object sender, EventArgs e )
    {
      OpenFileDialog of   = new OpenFileDialog();
			of.InitialDirectory = Application.StartupPath;
			of.CheckFileExists  = true;
      of.Filter           = "LogicImage board save data (*.logic_image_save)|*.logic_image_save";

      DialogResult res = of.ShowDialog();
      if ( res == DialogResult.OK )
      {
        game = CGameLoader.LoadGame( of.FileName );
        if ( game != null )
        {
          game.PrepareOwnerForm( this );     
        }
      }
    }

    void menu_SaveGame( object sender, EventArgs e )
    {
      if ( game != null )
      {
        SaveFileDialog of   = new SaveFileDialog();
  	    of.InitialDirectory = Application.StartupPath;
   			of.OverwritePrompt  = true;
        of.Filter           = "LogicImage board save data (*.logic_image_save)|*.logic_image_save";

        DialogResult res = of.ShowDialog();
        if ( res == DialogResult.OK )
        {
          CGameLoader.SaveGame( game, of.FileName );
          MessageBox.Show( "Game saved to "+of.FileName, "Info", MessageBoxButtons.OK, MessageBoxIcon.Information );
        }
      }
    }

    void menu_About( object sender, EventArgs e )
    {
      (new CAboutForm()).ShowDialog();
    }

    void menu_Exit( object sender, EventArgs e )
    {
      Close();
    }

    void menu_Undo( object sender, EventArgs e )
    {
      if ( game != null ) game.UndoMove();
    }

    void menu_Redo( object sender, EventArgs e )
    {
      if ( game != null ) game.RedoMove();
    }

    // -----------------------------------------------------------
    // events
    // -----------------------------------------------------------
    Point mmPoint;
    void this_MouseDown( object sender, MouseEventArgs e )
    {
      if ( game != null )
      {
        mmPoint = game.GetCellFromPoint( e.X, e.Y );
        game.SquareClicked( e.X, e.Y, 
                            (e.Button==MouseButtons.Left ? 
                             CGame.SquareClick.Forward : 
                             CGame.SquareClick.Backward) );
      }
    }
    void this_MouseMove( object sender, MouseEventArgs e )
    {
      if ( game != null && e.Button != MouseButtons.None )
      {
        Point mmNewPoint = game.GetCellFromPoint( e.X, e.Y );
        if ( mmNewPoint != mmPoint )
        {
          game.SquareClicked( e.X, e.Y, 
                              (e.Button==MouseButtons.Left ? 
                              CGame.SquareClick.Forward : 
                               CGame.SquareClick.Backward) );
          mmPoint = mmNewPoint;
        }
      }
    }

    // -----------------------------------------------------------
    // drawing
    // -----------------------------------------------------------
    protected override void OnPaint( PaintEventArgs e )
    {
      base.OnPaint( e );

      if ( game != null )
        game.PaintTheBoard( e.Graphics );
      else
      {
        using ( FontFamily ft = new FontFamily( "Tahoma" ) )
        {
          StringFormat sf = new StringFormat();
          sf.Alignment    = sf.LineAlignment = StringAlignment.Center;

          Rectangle r     = this.ClientRectangle;

          GraphicsPath gp = new GraphicsPath();
          gp.AddString( Title, ft, 1, 36, r, sf );

          e.Graphics.ScaleTransform( 1.0f, 3.0f );
          e.Graphics.TranslateTransform( 0, -r.Height/3 );
          e.Graphics.SmoothingMode = SmoothingMode.HighQuality;

          using ( LinearGradientBrush lgb = 
                  new LinearGradientBrush( r, Color.Red, Color.Blue, LinearGradientMode.BackwardDiagonal ) )
          {
            e.Graphics.FillPath( lgb, gp );
            using ( Pen p = new Pen( Color.Black, 2 ) )
            {
              e.Graphics.DrawPath( p, gp );
            }
          }
        }
      }
    }

  }
}