using System;
using System.IO;
using System.Drawing;
using System.Windows.Forms;
using System.Text;
using System.Xml;

namespace LogicImage
{
  public class CGameLoader
  {
    private CGameLoader() {}

    public static CGame NewGame( string sFileName )
    {
      try
      {
        XmlDocument xDoc = new XmlDocument();
        xDoc.Load( sFileName );

        XmlNode     xNode;
        XmlNodeList xList;

        // gridsize & gridmargin
        xNode = xDoc.SelectSingleNode( @"//LogicImage/GridSize" );
        int GridSize = int.Parse( xNode.InnerText );

        xNode = xDoc.SelectSingleNode( @"//LogicImage/GridMargin" );
        int GridMargin = int.Parse( xNode.InnerText );

        xNode = xDoc.SelectSingleNode( @"//LogicImage/CellSizeX" );
        int CellSizeX = int.Parse( xNode.InnerText );
        xNode = xDoc.SelectSingleNode( @"//LogicImage/CellSizeY" );
        int CellSizeY = int.Parse( xNode.InnerText );

        xNode = xDoc.SelectSingleNode( @"//LogicImage/Author" );
        string Author = xNode.InnerText;

        xNode = xDoc.SelectSingleNode( @"//LogicImage/Description" );
        string Description = xNode.InnerText;

        CGame game = new CGame( GridSize, GridMargin, 
                                CellSizeX, CellSizeY,
                                Author, Description );

        xList = xDoc.SelectNodes( @"//LogicImage/Row" );
        for ( int r=0; r<xList.Count; r++ )
          if ( xList[r].InnerText.Trim() != string.Empty ) 
          {         
            string[] rData = xList[r].InnerText.Split( new char[] { ',' } );
            for ( int i=0; i<rData.Length; i++ )
              game.RowData[ r, game.GridMargin-rData.Length+i ] = int.Parse( rData[i] );
          }
          
        xList = xDoc.SelectNodes( @"//LogicImage/Col" );
        for ( int c=0; c<xList.Count; c++ )
          if ( xList[c].InnerText.Trim() != string.Empty ) 
          {
             string[] cData = xList[c].InnerText.Split( new char[] { ',' } );
             for ( int i=0; i<cData.Length; i++ )
               game.ColData[ game.GridMargin-cData.Length+i, c ] = int.Parse( cData[i] );
          }

        return game;
      }
      catch( Exception ex )
      {
        MessageBox.Show( "Error loading image.\r\n"+ex.Message, "Error",
                         MessageBoxButtons.OK, MessageBoxIcon.Error );
        return null;
      }
    }

    public static CGame LoadGame( string sFileName )
    {
      CGame game = NewGame( sFileName );
      if ( game != null )
      {
        // load moves
        XmlDocument xDoc = new XmlDocument();
        xDoc.Load( sFileName );

        XmlNodeList xList;
        xList = xDoc.SelectNodes( @"//LogicImage/Move" );
        for ( int r=0; r<xList.Count; r++ )
        {
           string[] rS = xList[r].InnerText.Split( new char[] { ',' } );

           if ( rS.Length == 3 )
           {
             int x = int.Parse(rS[0]);
             int y = int.Parse(rS[1]);
             CGame.SquareClick click = (CGame.SquareClick)int.Parse(rS[2]);

             // "click the point"
             Rectangle rC = game.GetCellRect( x, y );
             game.SquareClicked( rC.X+1, rC.Y+1, click );
           }
        }
      }
      return game;
    }

    public static void SaveGame( CGame game, string sFileName )
    {
      if ( File.Exists( sFileName ) ) File.Delete( sFileName );

      XmlDocument xDoc = new XmlDocument();
      XmlDeclaration xDecl = xDoc.CreateXmlDeclaration( "1.0", "UTF-8", null );
      xDoc.AppendChild( xDecl );

      // main
      XmlElement xLI = xDoc.CreateElement( "LogicImage" );
      xDoc.AppendChild( xLI );

      // attributes
      XmlElement xNode;

      xNode = xDoc.CreateElement( "Version" );
      xNode.InnerText = "1.0";
      xLI.AppendChild( xNode );

      xNode = xDoc.CreateElement( "GridSize" );
      xNode.InnerText = game.GridSize.ToString();
      xLI.AppendChild( xNode );
      xNode = xDoc.CreateElement( "GridMargin" );
      xNode.InnerText = game.GridMargin.ToString();
      xLI.AppendChild( xNode );

      xNode = xDoc.CreateElement( "CellSizeX" );
      xNode.InnerText = game.CellSizeX.ToString();
      xLI.AppendChild( xNode );
      xNode = xDoc.CreateElement( "CellSizeY" );
      xNode.InnerText = game.CellSizeY.ToString();
      xLI.AppendChild( xNode );

      xNode = xDoc.CreateElement( "Author" );
      xNode.InnerText = game.Author.ToString();
      xLI.AppendChild( xNode );
      xNode = xDoc.CreateElement( "Description" );
      xNode.InnerText = game.Description.ToString();
      xLI.AppendChild( xNode );

      // Row
      for ( int r=0; r<game.GridSize; r++ )
      { 
        string sRow = string.Empty;
        for ( int c=0; c<game.GridMargin; c++ )
          if ( game.RowData[r,c]!=0 )
          {
            sRow += game.RowData[r,c].ToString();
            if ( c<game.GridMargin-1 )
              sRow+=",";
          }
        xNode = xDoc.CreateElement( "Row" );
        xNode.InnerText = sRow;
        xLI.AppendChild( xNode );
      }

      // Col
      for ( int c=0; c<game.GridSize; c++ )
      { 
        string sCol = string.Empty;
        for ( int r=0; r<game.GridMargin; r++ )
          if ( game.ColData[r,c]!=0 )
          {
            sCol += game.ColData[r,c].ToString();
            if ( r<game.GridMargin-1 )
              sCol+=",";
          }
        xNode = xDoc.CreateElement( "Col" );
        xNode.InnerText = sCol;
        xLI.AppendChild( xNode );
      }

      // Moves
      foreach ( CGame.Move move in game.Moves )
      {
        xNode = xDoc.CreateElement( "Move" );
        xNode.InnerText = string.Format( "{0},{1},{2}", move.X, move.Y, (int)move.Click );
        xLI.AppendChild( xNode );
      }
      
      // writer
      XmlTextWriter xmlW = new XmlTextWriter( sFileName, Encoding.UTF8 );
      xmlW.Formatting = Formatting.Indented;

      xDoc.WriteTo( xmlW );
      xmlW.Flush();
      xmlW.Close();
    }
  }
}