﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Drawing;
using System.Windows.Forms;

namespace FastTetris
{
    public class Board
    {
        public enum PieceMove { Left, Right, Up, Down, Rotate }

        const int MARGIN           = 5;
        const int SIZEX            = 10;
        const int SIZEY            = 40;
        const int DRAWINGBLOCKSIZE = 20;
        const int PIECESTARTX      = 10;

        bool[,]   squares;

        Piece     CurrentPiece;
        int       CurrentPieceX;
        int       CurrentPieceY;
        public int Score;

        public Board()
        {
            squares = new bool[MARGIN + SIZEX + MARGIN, MARGIN + SIZEY + MARGIN];

            // wypełnienie marginesu
            for ( int i=0; i < MARGIN + SIZEX + MARGIN; i++ )
            {
                for ( int j=0; j < MARGIN; j++ ) // góra
                {
                    squares[i, j] = true;
                    squares[i, j + SIZEY + MARGIN] = true;
                }
            }
            for ( int i=0; i < MARGIN; i++ )
            {
                for ( int j=0; j < MARGIN + SIZEY + MARGIN; j++ )
                {
                    squares[i, j] = true;
                    squares[i + SIZEX + MARGIN, j] = true;
                }
            }

            CurrentPiece  = Piece.GetRandomPiece();
            CurrentPieceX = PIECESTARTX;
            CurrentPieceY = MARGIN;

        }

        #region Drawing

        public Bitmap GetBitmap()
        {
            Bitmap _bitmap = new Bitmap( DRAWINGBLOCKSIZE * SIZEX, DRAWINGBLOCKSIZE * SIZEY );

            using ( Graphics g = Graphics.FromImage( _bitmap ) )
            {
                g.SmoothingMode = System.Drawing.Drawing2D.SmoothingMode.HighQuality;
                g.InterpolationMode = System.Drawing.Drawing2D.InterpolationMode.HighQualityBicubic;
                g.Clear( Color.White );

                // stan dotychczasowy
                for ( int i=0; i < SIZEX; i++ )
                    for ( int j=0; j < SIZEY; j++ )
                        if ( squares[MARGIN + i, MARGIN + j] )
                        {
                            g.FillRectangle( Brushes.Black, new Rectangle( new Point( i * DRAWINGBLOCKSIZE, j * DRAWINGBLOCKSIZE ), new Size( DRAWINGBLOCKSIZE, DRAWINGBLOCKSIZE ) ) );
                        }

                // klocek
                if ( CurrentPiece != null )
                {
                    for ( int i=0; i < Piece.SIZE; i++ )
                        for ( int j=0; j < Piece.SIZE; j++ )
                            if ( CurrentPiece.BlockData[i, j] )
                            {
                                g.FillRectangle( Brushes.Black, new Rectangle( new Point( ( CurrentPieceX - MARGIN + i ) * DRAWINGBLOCKSIZE, ( CurrentPieceY - MARGIN + j ) * DRAWINGBLOCKSIZE ), new Size( DRAWINGBLOCKSIZE, DRAWINGBLOCKSIZE ) ) );
                            }
                }

                // score
                StringFormat sf = new StringFormat();
                sf.Alignment = StringAlignment.Far;
                using ( Font f = new Font( "Tahoma", 12.0f, FontStyle.Regular ) )
                    g.DrawString( string.Format( "Score: {0}", this.Score ), f, Brushes.Red, new Point( _bitmap.Width, 10 ), sf );
            }

            return _bitmap;
        }

        #endregion

        #region Game logic

        /// <summary>
        /// Moves a piece automatically a step down. 
        /// </summary>
        /// <returns></returns>
        internal bool AutoUpdate()
        {
            RemoveFullLines();

            if ( CurrentPiece != null )
            {
                int _newX = CurrentPieceX;
                int _newY = CurrentPieceY + 1;

                if ( IsLegalPosition( CurrentPiece, _newX, _newY ) )
                {
                    CurrentPieceY = _newY;
                }
                else
                {
                    AbsorbCurrentPiece();

                    CurrentPiece = Piece.GetRandomPiece();
                    CurrentPieceX = PIECESTARTX;
                    CurrentPieceY = MARGIN;

                    if ( !IsLegalPosition( CurrentPiece, CurrentPieceX, CurrentPieceY ) )
                        return false;
                }
            }

            return true;
        }

        #region Removing lines

        private void RemoveFullLines()
        {
            int FullLineNumber = -1;
            do
            {
                FullLineNumber = GetPossibleFullLine();
                if ( FullLineNumber >= 0 )
                    RemoveLine( FullLineNumber );
            } while ( FullLineNumber >= 0 );
        }

        private int GetPossibleFullLine()
        {
            for ( int j = 0; j < SIZEY; j++ )
            {
                bool isFull = true;
                for ( int i = 0; i < SIZEX; i++ )
                    isFull &= this.squares[MARGIN + i, MARGIN + j];

                if ( isFull )
                    return j + MARGIN;
            }

            return -1;
        }

        /// <summary>
        /// Wszystko z linii POWYŻEJ LineNumber trzeba przenieść o jeden w dół
        /// </summary>
        /// <param name="LineNumber"></param>
        /// <returns></returns>
        private void RemoveLine( int LineNumber )
        {
            for ( int j = LineNumber; j > MARGIN; j-- )
                for ( int i=MARGIN; i < SIZEX + MARGIN; i++ )
                    this.squares[i, j] = this.squares[i, j - 1];

            Score += 100;
        }

        #endregion

        /// <summary>
        /// Copy piece data to board's block data. Called only when piece becomes "solid", non-movable.
        /// </summary>
        private void AbsorbCurrentPiece()
        {
            if ( CurrentPiece != null )
                for ( int i=0; i < Piece.SIZE; i++ )
                    for ( int j=0; j < Piece.SIZE; j++ )
                        if ( CurrentPiece.BlockData[i, j] )
                            this.squares[i + CurrentPieceX, j + CurrentPieceY] = true;
        }

        /// <summary>
        /// Moves current piece into a new position, if possible
        /// </summary>
        /// <param name="Move"></param>
        internal void MoveCurrentPiece( PieceMove Move )
        {
            Piece _newPiece;
            int   _newX;
            int   _newY;

            _newPiece = CurrentPiece;
            _newX = CurrentPieceX;
            _newY = CurrentPieceY;
            if ( CurrentPiece != null )
            {
                switch ( Move )
                {
                    case PieceMove.Rotate :
                        _newPiece = Piece.Rotate( CurrentPiece ); break;
                    case PieceMove.Left :
                        _newX = CurrentPieceX - 1; break;
                    case PieceMove.Right:
                        _newX = CurrentPieceX + 1; break;
                    case PieceMove.Up:
                        _newY = CurrentPieceY - 1; break;
                    case PieceMove.Down:
                        _newY = CurrentPieceY + 1; break;
                    default :
                        break;
                }

                // move only if legal position
                if ( this.IsLegalPosition( _newPiece, _newX, _newY ) )
                {
                    CurrentPiece = _newPiece;
                    CurrentPieceX = _newX;
                    CurrentPieceY = _newY;
                }
            }
        }

        /// <summary>
        /// Is a given position "legal" in a sense that it does not overlap with non-empty cells?
        /// </summary>
        /// <param name="piece"></param>
        /// <param name="pieceX"></param>
        /// <param name="pieceY"></param>
        /// <returns></returns>
        public bool IsLegalPosition( Piece piece, int pieceX, int pieceY )
        {
            for ( int i=0; i < Piece.SIZE; i++ )
                for ( int j=0; j < Piece.SIZE; j++ )
                    if ( piece.BlockData[i, j] &&
                         this.squares[i + pieceX, j + pieceY]
                        )
                        return false;

            return true;
        }

        #endregion
    }
}
