import sys
import pygame
from pygame.locals import *
from math import sqrt

from libs import Point, Draw, PriorityQueue

########################################
###################	Heuristics #########
########################################
heuristics = ['manhattan', 'diagonal', 'euclides']
def manhattan(p1, p2):
	return abs(p1.x - p2.x) + abs(p1.y - p2.y)

def diagonal(p1, p2):
	return max(abs(p1.x - p2.x), abs(p1.y - p2.y))
	
def euclides(p1, p2):
	return sqrt((p1.x - p2.x)**2 + (p1.y - p2.y)**2)
########################################
###################	Cost function ######
########################################
def f(distances, heuristic, p, end, parent):
	# multiple heuristics by 10 (static fixed number) and add already computed distance
	return 10 * heuristic(p, end) + distances[p.to_tuple()]
	
########################################
###################	Distance function ##
########################################
def g(parent, p):
	if parent.x == p.x or parent.y == p.y:
		return 10
	return 14	
		
########################################
###################	Algorithm ##########
########################################
def a_star(heuristic, start, end, missing, size_x, size_y, x_res, y_res, grid_size):
	# initialize scene
	scene = Draw(x_res, y_res, grid_size, start, end, missing)
	scene.draw_scene()
	scene.refresh()
	
	# initialize structures
	opened = set() 
	closed = set() # already visited nodes
	parents = {} 
	distances = {} 
	pq = PriorityQueue()

	# start with starting point
	opened.add(start.to_tuple())
	pq.add(heuristic(start, end), start, start)
	parents[start.to_tuple()] = start.to_tuple()
	distances[start.to_tuple()] = 0;
	
	# main loop
	steps = 0
	while opened:
		# select node minimalizing cost function
		current = pq.get_top_priority()

		# redraw scene
		print("Press any key to continue...")
		while pygame.event.wait().type != KEYDOWN:
			pass
		path = []
		parent = current[1].to_tuple()
		while parent != parents[parent]:
			path.append(parent)
			parent = parents[parent]

		scene.clear()
		scene.draw_scene()
		scene.draw_path(path)
		scene.refresh()
		
		# check if algorithm achieved ending point		
		if(current[1] == end):
			# print cost of found path
			print "Path of cost %s found in %s steps" % (distances[current[1].to_tuple()], steps)
			# construct found path
			path = [start]
			parent = parents[end.to_tuple()]
			path.append(parent)
			while parent != parents[parent]:
				path.append(parent)
				parent = parents[parent]
			path.append(end)
			# print nodes of found path
			for node in path:
				print node
			return
					
		# mark current node as visited
		closed.add(current[1].to_tuple())
		opened.remove(current[1].to_tuple())
		
		# for each neighbor of current point
		for i in range(-1, 2):
			for j in range(-1, 2):
				if i != 0 or j != 0:
					p = Point(current[1].x + i, current[1].y + j)
					# check if neighbor is in field
					if p.x >= 0 and p.x < size_x and p.y >= 0 and p.y < size_y:
						# check if neighbor isn't missing node and wasn't visited earlier
						if p not in missing and p.to_tuple() not in closed:
							# check if neighbor is visited for the first time
							if p.to_tuple() not in opened:
								opened.add(p.to_tuple())
								# computing distance to neighbor
								distances[p.to_tuple()] = distances[current[1].to_tuple()] + g(current[1], p)
								# push neighbor to priority queue
								pq.add(f(distances, heuristic, p, end, current[1]), p, current[1])
								# mark current node as a parent for neighbor
								parents[p.to_tuple()] = current[1].to_tuple()
							else:
								# check if new path to neighbor have lower cost than the current path
								if f(distances, heuristic, p, end, current[1]) < pq.get_value(p):
									# recomputing distance to neighbor
									distances[p.to_tuple()] = distances[current[1].to_tuple()] + g(current[1], p)
									# change priority of neighbor in priority queue
									pq.change_priority(p, f(distances, heuristic, p, end, current[1]), current[1])
									# mark current node as a parent for neighbor
									parents[p.to_tuple()] = current[1].to_tuple()
								
		steps += 1
	# if there isn't not visited node
	print "Path does not exists" 
							
########################################
###################	Test cases #########
########################################
if __name__ == "__main__":
	if len(sys.argv) < 2:
		sys.exit('Usage: %s heuristic-name' % sys.argv[0])

	heuristic_name = sys.argv[1]
	if not heuristic_name in heuristics:
		sys.exit('Unknown heuristic: %s' % sys.argv[1])
	
	heuristic = eval(heuristic_name)
	size_x = int(raw_input())
	size_y = int(raw_input())
	resolution_x = int(raw_input())
	resolution_y = int(raw_input())
	grid_size = int(raw_input())
	start = eval(raw_input())
	start = Point(start[0], start[1])
	end = eval(raw_input())
	end = Point(end[0], end[1])
	num_of_missing_points = int(raw_input())
	missing = []
	for i in range(0, num_of_missing_points):
		point = eval(raw_input())
		missing.append(Point(point[0], point[1]))
	a_star(heuristic, start, end, missing, size_x, size_y, resolution_x, resolution_y, grid_size)

	print("Press any key to EXIT...")
	while pygame.event.wait().type != KEYDOWN: 
		pass


