import itertools
import heapq
import pygame

# Drawing
class Point(object):
	
	def __init__(self, x, y):
		self.x = x
		self.y = y
	
	def to_tuple(self):
		return (self.x, self.y)
		
	def __eq__(self, other):
		return self.x == other.x and self.y == other.y

	def __repr__(self):
		return "(%s, %s)" % (self.x, self.y)
		
class Draw:

	def __init__(self, x_res, y_res, grid_size, start, end, closed):
		self.colors = {
			'red': (255, 0, 0),
			'green': (0, 255, 0),
			'blue': (0, 0, 255),
			'gray': (49, 79, 79),
			'yellow': (250, 250, 10),
			'black': (0, 0, 0)
		}
		self.x_res = x_res
		self.y_res = y_res
		self.grid_size = grid_size
		self.start = start
		self.end = end
		self.closed = closed
		#initialize window
		pygame.init() 
		self.screen = pygame.display.set_mode((x_res, y_res)) 		
		
	def draw_scene(self):
		self.set_start_point(self.start)
		self.set_end_point(self.end)
		self.set_closed_points(self.closed)
		self.set_grid()
		
	def draw_path(self, path):
		for p in path:
			pygame.draw.rect(self.screen, self.colors['yellow'], (p[0] * self.grid_size, p[1] * self.grid_size, self.grid_size, self.grid_size), 0)
		
	def set_grid(self):
		for x in range(self.grid_size, self.x_res, self.grid_size):
			pygame.draw.line(self.screen, self.colors['gray'], (x, 0), (x, self.y_res), 1)
		
		for y in range(self.grid_size, self.x_res, self.grid_size):
			pygame.draw.line(self.screen, self.colors['gray'], (0, y), (self.x_res, y), 1)
		
	def set_start_point(self, point):
		self.start = point
		pygame.draw.rect(self.screen, self.colors['green'], (self.start.x * self.grid_size, self.start.y * self.grid_size, self.grid_size, self.grid_size), 0)
		pygame.display.update()

	def set_end_point(self, point):
		self.end = point
		pygame.draw.rect(self.screen, self.colors['red'], (self.end.x * self.grid_size, self.end.y * self.grid_size, self.grid_size, self.grid_size), 0)
		pygame.display.update()
		
	def set_closed_points(self, points):
		self.closed = points

		for p in self.closed:
			pygame.draw.rect(self.screen, self.colors['blue'], (p.x * self.grid_size, p.y * self.grid_size, self.grid_size, self.grid_size), 0)

	def refresh(self):
		pygame.display.flip()
		
	def clear(self):
		self.screen.fill(self.colors['black'])

class PriorityQueue:
	
	def __init__(self):
		self.pq = []                         # the priority queue list
		self.counter = itertools.count(1)    # unique sequence count
		self.bag = {}                # mapping of tasks to entries
		self.closed = set()
		self.INVALID = 0  
		
	def add(self, priority, point, parent, count=None):
		if not count:
			count = next(self.counter)
		entry = [priority, count, point, parent]
		self.bag[point.to_tuple()] = entry
		heapq.heappush(self.pq, entry)
		
	def get_top_priority(self):
		while True:
			priority, count, point, parent = heapq.heappop(self.pq)
			del self.bag[point.to_tuple()]
			if count != self.INVALID:
				return (priority, point, parent)
				
	def delete(self, point):
		entry = self.bag[point.to_tuple()]
		entry[1] = self.INVALID
		
	def change_priority(self, point, priority, parent):
		entry = self.bag[point.to_tuple()]
		self.add(priority, point, parent, entry[1])
		entry[1] = self.INVALID
	
	def get_value(self, point):
		entry = self.bag[point.to_tuple()]
		return entry[0]

