Require Import Utf8.
Require Import Lang.

Definition term_rel := ∀ V : Set, term V → term V → Prop.

Definition id (R : term_rel) : term_rel := R.

Notation "R ⊆ S" := (∀ (V : Set) (t₁ t₂ : term V), R V t₁ t₂ → S V t₁ t₂)
  (at level 70).

Definition trf_sum (f g : term_rel → term_rel) : term_rel → term_rel :=
  λ R V t₁ t₂, f R V t₁ t₂ ∨ g R V t₁ t₂.
Notation "f ∪ g" := (trf_sum f g) (at level 45).

Definition trf_comp (f g : term_rel → term_rel) : term_rel → term_rel :=
  λ R, f (g R).
Notation "f ◦ g" := (trf_comp f g) (at level 45).

Fixpoint iter (n : nat) (f : term_rel → term_rel) : term_rel → term_rel :=
  match n with
  | 0   => id
  | S n => iter n f ∪ (f ◦ iter n f)
  end.

Notation "f ^ n" := (iter n f).

Definition iter_omega (f : term_rel → term_rel) (R : term_rel)
    (V : Set) (t₁ t₂ : term V) : Prop :=
  ∃ n, (f ^ n) R V t₁ t₂.

Notation "f ^ω" := (iter_omega f) (at level 45).

Definition monotone (f : term_rel → term_rel) : Prop :=
  ∀ R S : term_rel, R ⊆ S → f R ⊆ f S.

Definition finite (f : term_rel → term_rel) : Prop :=
  ∀ g : term_rel → term_rel, (∀ R, f R ⊆ g R) →
  ∀ R, f (g ^ω R) ⊆ g ^ω R.

Lemma rel_sub_trans (R Q S : term_rel) :
  R ⊆ Q → Q ⊆ S → R ⊆ S.
Proof.
auto.
Qed.

Lemma rel_sub_refl (R : term_rel) : R ⊆ R.
Proof.
auto.
Qed.

Lemma sum_subset (R S : term_rel) (f g : term_rel → term_rel) :
  f R ⊆ S → g R ⊆ S → (f ∪ g) R ⊆ S.
Proof.
unfold trf_sum; firstorder.
Qed.

Lemma sum_sub_l (R S : term_rel) (f g : term_rel → term_rel) :
  R ⊆ f S → R ⊆ (f ∪ g) S.
Proof.
unfold trf_sum; firstorder.
Qed.

Lemma sum_sub_r (R S : term_rel) (f g : term_rel → term_rel) :
  R ⊆ g S → R ⊆ (f ∪ g) S.
Proof.
unfold trf_sum; firstorder.
Qed.

Lemma iter_monotone_n (f : term_rel → term_rel) (R : term_rel) (n m : nat) :
  n ≤ m → (f ^ n) R ⊆ (f ^ m) R.
Proof.
induction 1; [ auto | ].
left; auto.
Qed.

Lemma iter_monotone_f (n : nat) (f g : term_rel → term_rel) :
  monotone g →
  (∀ R, f R ⊆ g R) → (∀ R, (f ^ n) R ⊆ (g ^ n) R).
Proof.
intros Hg Hf; induction n; intro R; simpl; [ auto | ].
apply sum_subset.
+ apply sum_sub_l, IHn.
+ apply sum_sub_r.
  eapply rel_sub_trans; [ apply Hf | ].
  apply Hg; auto.
Qed.

Lemma omega_monotone_f (f g : term_rel → term_rel) :
  monotone g →
  (∀ R, f R ⊆ g R) → (∀ R, (f ^ω) R ⊆ (g ^ω) R).
Proof.
intros Hg Hf R V t₁ t₂ Ht; destruct Ht as [ n Ht ].
apply (iter_monotone_f n f g) in Ht; try assumption.
exists n; auto.
Qed.

Lemma omega_limit (f : term_rel → term_rel) (R S : term_rel) :
  (∀ n, (f ^ n) R ⊆ S) → (f ^ω) R ⊆ S.
Proof.
intros Hn V t₁ t₂ Ht; destruct Ht; eauto.
Qed.

Lemma omega_id (f : term_rel → term_rel) (R : term_rel) :
  R ⊆ f ^ω R.
Proof.
exists 0; assumption.
Qed.

Lemma omega_one (f : term_rel → term_rel) (R : term_rel) :
  f R ⊆ f ^ω R.
Proof.
exists 1; right; assumption.
Qed.

Lemma omega_comp_omega (f : term_rel → term_rel) (R : term_rel) :
  monotone f → finite f → f ^ω (f ^ω R) ⊆ f ^ω R.
Proof.
intros Hmon Hfin V t₁ t₂ Ht; destruct Ht as [ n Ht ].
generalize R V t₁ t₂ Ht; clear R V t₁ t₂ Ht.
induction n; simpl; [ auto | intro R ].
apply sum_subset; [ auto | ].
eapply rel_sub_trans.
+ apply Hmon, IHn.
+ eapply Hfin; auto.
Qed.

Lemma sum_monotone (f g : term_rel → term_rel) :
  monotone f → monotone g → monotone (f ∪ g).
Proof.
intros Hf Hg R S HRS; unfold trf_sum; firstorder.
Qed.

Lemma comp_monotone (f g : term_rel → term_rel) :
  monotone f → monotone g → monotone (f ◦ g).
Proof.
intros Hf Hg R S HRS; apply Hf, Hg; assumption.
Qed.

Lemma iter_monotone (n : nat) (f : term_rel → term_rel) :
  monotone f → monotone (f ^ n).
Proof.
intro Hf; induction n; simpl.
+ unfold monotone; auto.
+ apply sum_monotone; [ assumption | ].
  apply comp_monotone; assumption.
Qed.

Lemma omega_monotone (f : term_rel → term_rel) :
  monotone f → monotone (f ^ω).
Proof.
intros Hf R S HRS V t₁ t₂ HR.
destruct HR as [ n HR ].
eapply iter_monotone in HR; try eassumption.
exists n; assumption.
Qed.

Lemma sum_finite (f g : term_rel → term_rel) :
  finite f → finite g → finite (f ∪ g).
Proof.
intros Hf Hg h Hh R V t₁ t₂ [ HR | HR ].
+ apply Hf; [ | assumption ].
  intros; apply Hh; left; assumption.
+ apply Hg; [ | assumption ].
  intros; apply Hh; right; assumption.
Qed.