Require Import Utf8.

Inductive inc (V : Set) : Set :=
| VZ : inc V
| VS : V → inc V
.

Arguments VZ [V].
Arguments VS [V] _.

Inductive term (V : Set) : Set :=
| tm_val : value V → term V
| tm_app : term V → term V → term V
| tm_rst : term V → term V
with value (V : Set) : Set :=
| val_var : V → value V
| val_lam : term (inc V) → value V
| val_sft : value V
.

Coercion tm_val : value >-> term.

Arguments tm_val [V] _.
Arguments tm_app [V] _ _.
Arguments tm_rst [V] _.
Arguments val_var [V] _.
Arguments val_lam [V] _.
Arguments val_sft [V].

Notation term0  := (term Empty_set).
Notation value0 := (value Empty_set).

Inductive pure_term {V : Set} : term V → Prop :=
| pt_val : ∀ v : value V, pure_term v
| pt_rst : ∀ t : term V,  pure_term (tm_rst t)
.

Inductive ectx (V : Set) : Set :=
| ectx_mt   : ectx V
| ectx_app1 : ectx V → term V → ectx V
| ectx_app2 : value V → ectx V → ectx V
.

Arguments ectx_mt   [V].
Arguments ectx_app1 [V] _ _.
Arguments ectx_app2 [V] _ _.

Inductive mctx (V : Set) : Set :=
| mctx_mt   : mctx V
| mctx_app1 : mctx V → term V → mctx V
| mctx_app2 : value V → mctx V → mctx V
| mctx_rst  : mctx V → mctx V
.

Arguments mctx_mt   [V].
Arguments mctx_app1 [V] _ _.
Arguments mctx_app2 [V] _ _.
Arguments mctx_rst  [V] _.

Reserved Notation "E '$[' t ']'" (at level 40).

Fixpoint eplug {V : Set} (E : ectx V) (t : term V) : term V :=
  match E with
  | ectx_mt       => t
  | ectx_app1 E s => tm_app (E $[ t ]) s
  | ectx_app2 v E => tm_app v (E $[ t ])
  end
where "E $[ t ]" := (@eplug _ E t).

Fixpoint ecomp {V : Set} (E₁ E₂ : ectx V) : ectx V :=
  match E₁ with
  | ectx_mt        => E₂
  | ectx_app1 E₁ t => ectx_app1 (ecomp E₁ E₂) t
  | ectx_app2 v E₁ => ectx_app2 v (ecomp E₁ E₂)
  end.

Fixpoint mctx_pure {V : Set} (E : ectx V) : mctx V :=
  match E with
  | ectx_mt       => mctx_mt
  | ectx_app1 E t => mctx_app1 (mctx_pure E) t
  | ectx_app2 v E => mctx_app2 v (mctx_pure E)
  end.

Coercion mctx_pure : ectx >-> mctx.

Reserved Notation "F '$$[' t ']'" (at level 40).

Fixpoint mplug {V : Set} (F : mctx V) (t : term V) : term V :=
  match F with
  | mctx_mt       => t
  | mctx_app1 F s => tm_app (F $$[ t ]) s
  | mctx_app2 v F => tm_app v (F $$[ t ])
  | mctx_rst  F   => tm_rst (F $$[ t ])
  end
where "F $$[ t ]" := (@mplug _ F t).

Fixpoint mcomp {V : Set} (F₁ F₂ : mctx V) : mctx V :=
  match F₁ with
  | mctx_mt        => F₂
  | mctx_app1 F₁ t => mctx_app1 (mcomp F₁ F₂) t
  | mctx_app2 v F₁ => mctx_app2 v (mcomp F₁ F₂)
  | mctx_rst F₁    => mctx_rst (mcomp F₁ F₂)
  end.

Definition inc_map {A B : Set} (f : A → B) (x : inc A) : inc B :=
  match x with
  | VZ   => VZ
  | VS y => VS (f y)
  end.

Fixpoint tmap {A B : Set} (f : A → B) (t : term A) : term B :=
  match t with
  | tm_val v   => vmap f v
  | tm_app t s => tm_app (tmap f t) (tmap f s)
  | tm_rst t   => tm_rst (tmap f t)
  end
with vmap {A B : Set} (f : A → B) (v : value A) : value B :=
  match v with
  | val_var x => val_var (f x)
  | val_lam t => val_lam (tmap (inc_map f) t)
  | val_sft   => val_sft
  end.

Fixpoint emap {A B : Set} (f : A → B) (E : ectx A) : ectx B :=
  match E with
  | ectx_mt       => ectx_mt
  | ectx_app1 E t => ectx_app1 (emap f E) (tmap f t)
  | ectx_app2 v E => ectx_app2 (vmap f v) (emap f E)
  end.

Fixpoint mmap {A B : Set} (f : A → B) (F : mctx A) : mctx B :=
  match F with
  | mctx_mt       => mctx_mt
  | mctx_app1 F t => mctx_app1 (mmap f F) (tmap f t)
  | mctx_app2 v F => mctx_app2 (vmap f v) (mmap f F)
  | mctx_rst F    => mctx_rst (mmap f F)
  end.

Notation tshift := (tmap (@VS _)).
Notation vshift := (vmap (@VS _)).
Notation eshift := (emap (@VS _)).
Notation mshift := (mmap (@VS _)).

Definition lift {A B : Set} (f : A → value B) (x : inc A) : value (inc B) :=
  match x with
  | VZ   => val_var VZ
  | VS y => vshift (f y)
  end.

Fixpoint tbind {A B : Set} (f : A → value B) (t : term A) : term B :=
  match t with
  | tm_val v   => vbind f v
  | tm_app t s => tm_app (tbind f t) (tbind f s)
  | tm_rst t   => tm_rst (tbind f t)
  end
with vbind {A B : Set} (f : A → value B) (v : value A) : value B :=
  match v with
  | val_var x => f x
  | val_lam t => val_lam (tbind (lift f) t)
  | val_sft   => val_sft
  end.

Fixpoint ebind {A B : Set} (f : A → value B) (E : ectx A) : ectx B :=
  match E with
  | ectx_mt       => ectx_mt
  | ectx_app1 E t => ectx_app1 (ebind f E) (tbind f t)
  | ectx_app2 v E => ectx_app2 (vbind f v) (ebind f E)
  end.

Fixpoint mbind {A B : Set} (f : A → value B) (F : mctx A) : mctx B :=
  match F with
  | mctx_mt       => mctx_mt
  | mctx_app1 F t => mctx_app1 (mbind f F) (tbind f t)
  | mctx_app2 v F => mctx_app2 (vbind f v) (mbind f F)
  | mctx_rst  F   => mctx_rst (mbind f F)
  end.

Definition subst_func {A : Set} (v : value A) (x : inc A) : value A :=
  match x with
  | VZ   => v
  | VS y => val_var y
  end.

Notation "t '{t↦' v '}'" := (@tbind _ _ (@subst_func _ v) t) (at level 40).
Notation "w '{v↦' v '}'" := (@vbind _ _ (@subst_func _ v) w) (at level 40).
Notation "E '{E↦' v '}'" := (@ebind _ _ (@subst_func _ v) E) (at level 40).
Notation "F '{M↦' v '}'" := (@mbind _ _ (@subst_func _ v) F) (at level 40).

Inductive red {V : Set} : term V → term V → Prop :=
| red_beta : ∀ (t : term (inc V)) (v : value V),
    red (tm_app (val_lam t) v) (t {t↦ v})
| red_shift : ∀ (v : value V) (E : ectx V),
    red (tm_rst (E $[ tm_app val_sft v ]))
      (tm_rst (tm_app v (val_lam (tm_rst (eshift E $[ val_var VZ ])))))
| red_reset : ∀ (v : value V),
    red (tm_rst v) v
| red_app1 : ∀ (t t' s : term V),
    red t t' →
    red (tm_app t s) (tm_app t' s)
| red_app2 : ∀ (v : value V) (t t' : term V),
    red t t' →
    red (tm_app v t) (tm_app v t')
| red_rst : ∀ t t' : term V,
    red t t' →
    red (tm_rst t) (tm_rst t')
.

Require Import Relation_Operators.

Notation red_rtc := (@clos_refl_trans_1n _ (@red _)).