Require Import Utf8.

Inductive inc (V : Set) : Set :=
| VZ : inc V
| VS : V → inc V
.

Arguments VZ [V].
Arguments VS [V] _.

Inductive term (V : Set) : Set :=
| tm_val : value V → term V
| tm_app : term V → term V → term V
with value (V : Set) : Set :=
| val_var : V → value V
| val_lam : term (inc V) → value V
.

Coercion tm_val : value >-> term.

Arguments tm_val [V] _.
Arguments tm_app [V] _ _.
Arguments val_var [V] _.
Arguments val_lam [V] _.

Notation term0  := (term Empty_set).
Notation value0 := (value Empty_set).

Inductive ectx (V : Set) : Set :=
| ectx_mt   : ectx V
| ectx_app1 : ectx V → term V → ectx V
| ectx_app2 : value V → ectx V → ectx V
.

Arguments ectx_mt   [V].
Arguments ectx_app1 [V] _ _.
Arguments ectx_app2 [V] _ _.

Reserved Notation "E '$[' t ']'" (at level 40).

Fixpoint eplug {V : Set} (E : ectx V) (t : term V) : term V :=
  match E with
  | ectx_mt       => t
  | ectx_app1 E s => tm_app (E $[ t ]) s
  | ectx_app2 v E => tm_app v (E $[ t ])
  end
where "E $[ t ]" := (@eplug _ E t).

Fixpoint ecomp {V : Set} (E₁ E₂ : ectx V) : ectx V :=
  match E₁ with
  | ectx_mt        => E₂
  | ectx_app1 E₁ t => ectx_app1 (ecomp E₁ E₂) t
  | ectx_app2 v E₁ => ectx_app2 v (ecomp E₁ E₂)
  end.

Definition inc_map {A B : Set} (f : A → B) (x : inc A) : inc B :=
  match x with
  | VZ   => VZ
  | VS y => VS (f y)
  end.

Fixpoint tmap {A B : Set} (f : A → B) (t : term A) : term B :=
  match t with
  | tm_val v   => vmap f v
  | tm_app t s => tm_app (tmap f t) (tmap f s)
  end
with vmap {A B : Set} (f : A → B) (v : value A) : value B :=
  match v with
  | val_var x => val_var (f x)
  | val_lam t => val_lam (tmap (inc_map f) t)
  end.

Fixpoint emap {A B : Set} (f : A → B) (E : ectx A) : ectx B :=
  match E with
  | ectx_mt       => ectx_mt
  | ectx_app1 E t => ectx_app1 (emap f E) (tmap f t)
  | ectx_app2 v E => ectx_app2 (vmap f v) (emap f E)
  end.

Notation tshift := (tmap (@VS _)).
Notation vshift := (vmap (@VS _)).
Notation eshift := (emap (@VS _)).

Definition lift {A B : Set} (f : A → value B) (x : inc A) : value (inc B) :=
  match x with
  | VZ   => val_var VZ
  | VS y => vshift (f y)
  end.

Fixpoint tbind {A B : Set} (f : A → value B) (t : term A) : term B :=
  match t with
  | tm_val v   => vbind f v
  | tm_app t s => tm_app (tbind f t) (tbind f s)
  end
with vbind {A B : Set} (f : A → value B) (v : value A) : value B :=
  match v with
  | val_var x => f x
  | val_lam t => val_lam (tbind (lift f) t)
  end.

Fixpoint ebind {A B : Set} (f : A → value B) (E : ectx A) : ectx B :=
  match E with
  | ectx_mt       => ectx_mt
  | ectx_app1 E t => ectx_app1 (ebind f E) (tbind f t)
  | ectx_app2 v E => ectx_app2 (vbind f v) (ebind f E)
  end.

Definition subst_func {A : Set} (v : value A) (x : inc A) : value A :=
  match x with
  | VZ   => v
  | VS y => val_var y
  end.

Notation "t '{t↦' v '}'" := (@tbind _ _ (@subst_func _ v) t) (at level 40).
Notation "w '{v↦' v '}'" := (@vbind _ _ (@subst_func _ v) w) (at level 40).
Notation "E '{E↦' v '}'" := (@ebind _ _ (@subst_func _ v) E) (at level 40).

Inductive red {V : Set} : term V → term V → Prop :=
| red_beta : ∀ (t : term (inc V)) (v : value V),
    red (tm_app (val_lam t) v) (t {t↦ v})
| red_app1 : ∀ (t t' s : term V),
    red t t' →
    red (tm_app t s) (tm_app t' s)
| red_app2 : ∀ (v : value V) (t t' : term V),
    red t t' →
    red (tm_app v t) (tm_app v t')
.

Require Import Relation_Operators.

Notation red_rtc := (@clos_refl_trans_1n _ (@red _)).