#lang racket

;; Krzysztof Wrobel 2015-03-11
;; After Reynolds [1998]

(define (is-of-type exp type)
  (and (pair? exp) (eq? (car exp) type)))

;; application

(define (appl? exp)
  (pair? exp))
(define (mk-appl exp1 exp2)
  (list exp1 exp2))
(define (appl-opr exp)
  (car exp))
(define (appl-opnd exp)
  (cadr exp))

;; lambda abstraction

(define (lambda? exp)
  (is-of-type exp 'lambda))
(define (mk-lambda var exp)
  (list 'lambda var exp))
(define (lambda-param exp)
  (cadr exp))
(define (lambda-body exp)
  (caddr exp))

;; simple conditional

(define (cond? exp)
  (is-of-type exp 'cond))
(define (mk-cond exp1 exp2 exp3)
  (list 'cond exp1 exp2 exp3))
(define (cond-prem exp)
  (cadr exp))
(define (cond-conc exp)
  (caddr exp))
(define (cond-altr exp)
  (cadddr exp))

;; recursive let

(define (letrec? exp)
  (is-of-type exp 'letrec))
(define (mk-letrec var lambda exp)
  (list 'letrec var lambda exp))
(define (letrec-var exp)
  (cadr exp))
(define (letrec-exp exp)
  (caddr exp))
(define (letrec-body exp)
  (cadddr exp))

;; constants

(define (const? exp)
  (or (number? exp)
      (boolean? exp)))

;; variables

(define (var? exp)
  (symbol? exp))


;; Meta-Circular Interpreter

(define (eval exp env)
  (cond 
    [(const? exp) (evcon exp)]
    [(var? exp) (env exp)]
    [(lambda? exp) (evlambda exp env)]
    [(cond? exp) (if (eval (cond-prem exp) env)
                     (eval (cond-conc exp) env)
                     (eval (cond-altr exp) env))]
    [(letrec? exp) (letrec ([new-env (lambda (x) (if (eq? x (letrec-var exp))
                                                     (evlambda (letrec-exp exp) new-env)
                                                     (env x)))])
                     (eval (letrec-body exp) new-env))]
    [(appl? exp) ((eval (appl-opr exp) env) (eval (appl-opnd exp) env))]))


(define (evcon exp) exp)

(define (evlambda lexp env)
  (lambda (a) (eval (lambda-body lexp)
                    (ext (lambda-param lexp) a env))))

(define (ext var val env)
  (lambda (x) (if (eq? x var)
                  val
                  (env x))))

;; Initial environment

(define (initenv x)
  (cond
    [(eq? x 'succ) (lambda (a) (+ a 1))]
    [(eq? x 'equal) (lambda (a) (lambda (b) (eq? a b)))]))

;; interpret!

(define (interpret exp) (eval exp initenv))



;; examples


(interpret '(succ 1))

(define minus-ex '(letrec minus-aux
                    (lambda x
                      (lambda y
                        (lambda acc
                          (cond ((equal x) y)
                                acc
                                (((minus-aux x) (succ y)) (succ acc))))))
                    (((minus-aux 10) 7) 0)))

(interpret minus-ex)