#lang racket

;; Krzysztof Wrobel 2015-03-11
;; After Reynolds [1998]

(define (is-of-type exp type)
  (and (pair? exp) (eq? (car exp) type)))

;; application

(define (appl? exp)
  (pair? exp))
(define (mk-appl exp1 exp2)
  (list exp1 exp2))
(define (appl-opr exp)
  (car exp))
(define (appl-opnd exp)
  (cadr exp))

;; lambda abstraction

(define (lambda? exp)
  (is-of-type exp 'lambda))
(define (mk-lambda var exp)
  (list 'lambda var exp))
(define (lambda-param exp)
  (cadr exp))
(define (lambda-body exp)
  (caddr exp))

;; simple conditional

(define (cond? exp)
  (is-of-type exp 'cond))
(define (mk-cond exp1 exp2 exp3)
  (list 'cond exp1 exp2 exp3))
(define (cond-prem exp)
  (cadr exp))
(define (cond-conc exp)
  (caddr exp))
(define (cond-altr exp)
  (cadddr exp))

;; recursive let

(define (letrec? exp)
  (is-of-type exp 'letrec))
(define (mk-letrec var lambda exp)
  (list 'letrec var lambda exp))
(define (letrec-var exp)
  (cadr exp))
(define (letrec-exp exp)
  (caddr exp))
(define (letrec-body exp)
  (cadddr exp))

;; CLOSURES
;; lambda closure

(define (closr? exp)
  (is-of-type exp 'closr))
(define (mk-closr lambda env)
  (list 'closr lambda env))
(define (closr-lam exp)
  (cadr exp))
(define (closr-env exp)
  (caddr exp))

;; succ closure

(define (sc? exp)
  (is-of-type exp 'sc))
(define (mk-sc)
  (list 'sc))

;; closure of outer lambda in equal

(define (eq1? exp)
  (is-of-type exp 'eq1))
(define (mk-eq1)
  (list 'eq1))

;; closure of inner lambda in equal

(define (eq2? exp)
  (is-of-type exp 'eq2))
(define (mk-eq2 val)
  (list 'eq2 val))
(define (eq2-val exp)
  (cadr exp))

;; ENVIRONMENT
;; initial environment

(define (init-env? exp)
  (is-of-type exp 'init-env))
(define (mk-init-env)
  (list 'init-env))

;; extended environment

(define (ext-env? exp)
  (is-of-type exp 'ext-env))
(define (mk-ext-env var val env)
  (list 'ext-env var val env))
(define (ext-env-var exp)
  (cadr exp))
(define (ext-env-val exp)
  (caddr exp))
(define (ext-env-old exp)
  (cadddr exp))

;; recursive environment

(define (rec-env? exp)
  (is-of-type exp 'rec-env))
(define (mk-rec-env letrec env)
  (list 'rec-env letrec env))
(define (rec-env-letrec exp)
  (cadr exp))
(define (rec-env-old exp)
  (caddr exp))

;; constants

(define (const? exp)
  (or (number? exp)
      (boolean? exp)))

;; variables

(define (var? exp)
  (symbol? exp))


;; Interpreter defined using only first-order functions

(define (eval exp env)
  (cond 
    [(const? exp) (evcon exp)]
    [(var? exp) (get env exp)]
    [(lambda? exp) (mk-closr exp env)]
    [(cond? exp) (if (eval (cond-prem exp) env)
                     (eval (cond-conc exp) env)
                     (eval (cond-altr exp) env))]
    [(letrec? exp) (eval (letrec-body exp) (mk-rec-env exp env))]
    [(appl? exp) (apply (eval (appl-opr exp) env)
                        (eval (appl-opnd exp) env))]))


(define (apply proc arg)
  (cond
    [(closr? proc) (eval (lambda-body (closr-lam proc))
                         (mk-ext-env (lambda-param (closr-lam proc))
                                     arg
                                     (closr-env proc)))]
    [(sc? proc) (+ arg 1)]
    [(eq1? proc) (mk-eq2 arg)]
    [(eq2? proc) (eq? (eq2-val proc) arg)]))

(define (get env var)
  (cond
    [(init-env? env) (cond
                       [(eq? var 'succ) (mk-sc)]
                       [(eq? var 'equal) (mk-eq1)])]
    [(ext-env? env) (if (eq? var (ext-env-var env))
                        (ext-env-val env)
                        (get (ext-env-old env) var))]
    [(rec-env? env) (if (eq? var (letrec-var (rec-env-letrec env)))
                        (mk-closr (letrec-exp (rec-env-letrec env)) env)
                        (get (rec-env-old env) var))]))

(define (evcon exp) exp)

;; interpret!

(define (interpret exp) (eval exp (mk-init-env)))



;; examples


(interpret '(succ 1))

(define minus-ex '(letrec minus-aux
                    (lambda x
                      (lambda y
                        (lambda acc
                          (cond ((equal x) y)
                                acc
                                (((minus-aux x) (succ y)) (succ acc))))))
                    (((minus-aux 10) 7) 0)))

(interpret minus-ex)