module RegionsDeep where

type Point = (Double, Double)
type Vector = (Double, Double)
data Region = Empty
            | Circle
            | Square
            | Scale Vector Region
            | Translate Vector Region
            | Outside Region
            | Intersect Region Region
            | Union Region Region

inRegion :: Point -> Region -> Bool
distance :: Point -> Point -> Double
empty :: Region
circle :: Region
square :: Region
outside :: Region -> Region
scale :: Vector -> Region -> Region
translate :: Vector -> Region -> Region
(/\) :: Region -> Region -> Region
(\/) :: Region -> Region -> Region

distance (x1, y1) (x2, y2) = sqrt $ (x1 - x2) ** 2 + (y1 - y2) ** 2

inRegion p Empty = False
inRegion p Circle = distance p (0,0) <= 1
inRegion (x, y) Square = abs x <= 1 && abs y <= 1
inRegion p (Outside r) = not $ inRegion p r
inRegion (x, y) (Scale (sx, sy) r) = inRegion (x / sx, y / sy) r
inRegion (x, y) (Translate (sx, sy) r) = inRegion (x - sx, y - sy) r
inRegion p (Intersect r1 r2) = inRegion p r1 && inRegion p r2
inRegion p (Union r1 r2) = inRegion p r1 || inRegion p r2

empty = Empty
circle = Circle
square = Square
outside = Outside
scale = Scale
translate = Translate
(/\) = Intersect
(\/) = Union

circleAt :: Double -> Point -> Region
circleAt r center = translate center
                        $ scale (r, r) circle

circleRow :: Integer -> Region
circleRow n = foldr (\/) empty
                    [circleAt 1 (fromInteger x,0) | x <- [0..n-1]]

main = do
    let p = (0.0, 0.0) :: Point
    putStrLn $ show $
        p `inRegion` empty
    putStrLn $ show $
        (1.0, 0.0) `inRegion` circle
    putStrLn $ show $
        (1.5, 1.0) `inRegion` circle
    putStrLn $ show $
        (1.5, 1.0) `inRegion` (scale (2.0, 3.0) circle)
    putStrLn $ show $
        (1.0, 1.0) `inRegion` square
    putStrLn $ show $
        (1.0, 1.0) `inRegion` (translate (2.0, 3.0) square)
    putStrLn $ show $
        (0.0, 0.0) `inRegion` (square /\ (translate (1.0, 0.0) circle))
    putStrLn $ show $
        (1.0, 0.0) `inRegion` ((translate (-2.0, -5.0) square) \/ (scale (0.1, 0.2) circle))
    putStrLn $ show $
        (10, -20.0) `inRegion` (outside $ circleRow 100)

