/*
 * Decompiled with CFR 0.152.
 */
package math.matrices;

import java.util.Arrays;
import math.matrices.DimensionException;
import math.matrices.NotPositiveDefiniteMatrixException;
import math.matrices.UninvertibleMatrixException;
import math.matrices.Vector;
import math.utils.Numerics;

public class Matrix {
    private final double[][] fields;
    private final int rows;
    private final int cols;

    public Matrix(int rows, int cols) {
        this.rows = rows;
        this.cols = cols;
        this.fields = new double[rows][cols];
        for (int i = 0; i < Math.min(rows, cols); ++i) {
            this.fields[i][i] = 1.0;
        }
        this.ensureDimensionOK(rows, cols);
    }

    private void ensureDimensionOK(int rows, int cols) {
        if (rows <= 0) {
            throw new DimensionException("Number of rows must be positive, rows = " + rows);
        }
        if (cols <= 0) {
            throw new DimensionException("Number of columns must be positive, cols = " + cols);
        }
    }

    public Matrix(double[][] fields) {
        this.ensureFieldsOK(fields);
        this.rows = fields.length;
        this.cols = fields[0].length;
        this.fields = new double[this.rows][this.cols];
        this.copy(fields);
    }

    private void copy(double[][] fields) {
        for (int i = 0; i < this.rows; ++i) {
            for (int j = 0; j < this.cols; ++j) {
                this.fields[i][j] = fields[i][j];
            }
        }
    }

    private void ensureFieldsOK(double[][] fields) {
        this.ensureNotNull(fields, "Given array is null");
        this.ensureHasRows(fields, "Matrix must have at least one row");
        this.ensureRowsNotNull(fields, "Rows of the array may not be null");
        this.ensureRectangleArray(fields, "Rows of the array must have the same size");
        this.ensureHasCols(fields, "Matrix must have at least one column");
    }

    private void ensureNotNull(Object ob, String msg) {
        if (ob == null) {
            throw new IllegalArgumentException(msg);
        }
    }

    private void ensureHasRows(double[][] array, String msg) {
        if (array.length == 0) {
            throw new IllegalArgumentException(msg);
        }
    }

    private void ensureRowsNotNull(double[][] array, String msg) {
        for (int i = 0; i < array.length; ++i) {
            if (array[i] != null) continue;
            throw new IllegalArgumentException(msg);
        }
    }

    private void ensureRectangleArray(double[][] array, String msg) {
        int l = array[0].length;
        for (int i = 1; i < array.length; ++i) {
            if (array[i].length == l) continue;
            throw new IllegalArgumentException(msg);
        }
    }

    private void ensureHasCols(double[][] array, String msg) {
        if (array[0].length == 0) {
            throw new IllegalArgumentException(msg);
        }
    }

    public Matrix(Matrix m) {
        this.rows = m.rows;
        this.cols = m.cols;
        this.fields = new double[this.rows][this.cols];
        this.copy(m.fields);
    }

    public Matrix(Vector[] vectors) {
        this.ensureVectorListOK(vectors);
        this.rows = vectors[0].getRows();
        this.cols = vectors.length;
        this.fields = new double[this.rows][this.cols];
        for (int col = 1; col <= this.cols; ++col) {
            this.setCol(col, vectors[col - 1]);
        }
    }

    private void ensureVectorListOK(Vector[] vectors) {
        if (vectors == null) {
            throw new IllegalArgumentException("Array of vectors may not be null");
        }
        if (vectors.length == 0) {
            throw new IllegalArgumentException("Array of vectors may not be empty");
        }
        for (int i = 1; i < vectors.length; ++i) {
            if (vectors[i].getSize() == vectors[0].getSize()) continue;
            throw new IllegalArgumentException("Vectors must have the same size");
        }
    }

    protected Matrix(double[] vals) {
        this.rows = vals.length;
        this.cols = 1;
        this.fields = new double[this.rows][this.cols];
        for (int i = 0; i < this.rows; ++i) {
            this.fields[i][0] = vals[i];
        }
    }

    public Matrix transpose() {
        Matrix m = new Matrix(this.cols, this.rows);
        for (int i = 0; i < this.rows; ++i) {
            for (int j = 0; j < this.cols; ++j) {
                m.fields[j][i] = this.fields[i][j];
            }
        }
        return m;
    }

    public boolean isSquare() {
        return this.rows == this.cols;
    }

    public boolean isSymmetric() {
        return this.equals(this.transpose());
    }

    public boolean isPositiveDefinite() {
        if (!this.isSymmetric()) {
            return false;
        }
        try {
            this.cholesky();
            return true;
        }
        catch (NotPositiveDefiniteMatrixException ex) {
            return false;
        }
    }

    public boolean isCorrelationMatrix() {
        for (int row = 1; row <= this.rows; ++row) {
            if (Numerics.doublesEqual(this.get(row, row), 1.0)) continue;
            return false;
        }
        return this.isPositiveDefinite();
    }

    public Matrix cholesky() throws NotPositiveDefiniteMatrixException {
        if (!this.isSquare()) {
            throw new NotPositiveDefiniteMatrixException();
        }
        Matrix L = new Matrix(this.rows, this.rows);
        for (int row = 1; row <= this.rows; ++row) {
            this.calculateOneRowOfCholesky(row, L);
        }
        return L;
    }

    private void calculateOneRowOfCholesky(int row, Matrix L) throws NotPositiveDefiniteMatrixException {
        L.set(row, row, 0.0);
        this.calcNonDiagonalElementsOfCholesky(row, L);
        this.calcDiagonalElementOfCholesky(row, L);
    }

    private void calcNonDiagonalElementsOfCholesky(int row, Matrix L) {
        for (int col = 1; col < row; ++col) {
            this.calcNonDiagonalElementOfCholesky(row, col, L);
        }
    }

    private void calcNonDiagonalElementOfCholesky(int row, int col, Matrix L) {
        double sum = 0.0;
        for (int k = 1; k < col; ++k) {
            sum += L.get(row, k) * L.get(col, k);
        }
        L.set(row, col, (this.get(col, row) - sum) / L.get(col, col));
    }

    private void calcDiagonalElementOfCholesky(int row, Matrix L) throws NotPositiveDefiniteMatrixException {
        double sum = 0.0;
        for (int k = 1; k < row; ++k) {
            sum += L.get(row, k) * L.get(row, k);
        }
        double x = this.get(row, row) - sum;
        if (x < 0.0) {
            throw new NotPositiveDefiniteMatrixException();
        }
        L.set(row, row, Math.sqrt(x));
    }

    public Matrix add(Matrix other) throws DimensionException {
        this.ensureHasSameSize(other);
        Matrix res = new Matrix(this.rows, this.cols);
        for (int row = 1; row <= this.rows; ++row) {
            for (int col = 1; col <= this.cols; ++col) {
                res.set(row, col, this.get(row, col) + other.get(row, col));
            }
        }
        return res;
    }

    public Matrix times(double t) {
        Matrix res = new Matrix(this);
        for (int row = 1; row <= this.rows; ++row) {
            for (int col = 1; col <= this.cols; ++col) {
                res.set(row, col, t * res.get(row, col));
            }
        }
        return res;
    }

    private void ensureHasSameSize(Matrix other) {
        if (this.cols != other.cols || this.rows != other.rows) {
            throw new DimensionException("Matrix of size " + this.rows + "x" + this.cols + " cannot be added to matrix of size " + other.rows + "x" + other.cols);
        }
    }

    public Matrix mult(Matrix other) throws DimensionException {
        this.ensureCanBeMultiplied(other);
        try {
            Vector vec = (Vector)other;
            return this.multVector(vec);
        }
        catch (ClassCastException e) {
            return this.multMatrices(other);
        }
    }

    private void ensureCanBeMultiplied(Matrix other) {
        if (this.cols != other.rows) {
            throw new DimensionException("Multiplied matrices do not have suitable sizes (columns of left = " + this.cols + ", rows of right = " + other.rows + ")");
        }
    }

    private Matrix multMatrices(Matrix other) {
        int newRows = this.rows;
        int newCols = other.cols;
        Matrix res = new Matrix(newRows, newCols);
        for (int row = 1; row <= newRows; ++row) {
            for (int col = 1; col <= newCols; ++col) {
                res.set(row, col, this.multiplyRowAndCol(row, col, other));
            }
        }
        return res;
    }

    private Vector multVector(Vector vec) {
        Vector res = new Vector(this.rows);
        for (int row = 1; row <= this.rows; ++row) {
            res.set(row, this.multiplyRowAndCol(row, 1, vec));
        }
        return res;
    }

    private double multiplyRowAndCol(int row, int col, Matrix other) {
        double res = 0.0;
        for (int k = 1; k <= this.cols; ++k) {
            res += this.get(row, k) * other.get(k, col);
        }
        return res;
    }

    public Vector mult(Vector vec) throws DimensionException {
        return (Vector)this.mult((Matrix)vec);
    }

    public int hashCode() {
        int prime = 31;
        int result = 1;
        result = 31 * result + this.cols;
        result = 31 * result + Arrays.hashCode((Object[])this.fields);
        result = 31 * result + this.rows;
        return result;
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (this.getClass() != obj.getClass()) {
            return false;
        }
        Matrix other = (Matrix)obj;
        if (this.cols != other.cols) {
            return false;
        }
        if (this.rows != other.rows) {
            return false;
        }
        return this.properEquals(other);
    }

    private boolean properEquals(Matrix other) {
        for (int i = 0; i < this.rows; ++i) {
            for (int j = 0; j < this.cols; ++j) {
                if (Numerics.doublesEqual(this.fields[i][j], other.fields[i][j])) continue;
                return false;
            }
        }
        return true;
    }

    public final double get(int row, int col) {
        this.ensureIndicesOK(row, col);
        return this.fields[row - 1][col - 1];
    }

    protected void ensureIndicesOK(int row, int col) throws DimensionException {
        if (row <= 0 || row > this.rows) {
            throw new DimensionException("Invalid row index: " + row);
        }
        if (col <= 0 || col > this.cols) {
            throw new DimensionException("Invalid col index: " + col);
        }
    }

    public final void set(int row, int col, double val) {
        this.ensureIndicesOK(row, col);
        this.fields[row - 1][col - 1] = val;
    }

    public int getRows() {
        return this.rows;
    }

    public int getCols() {
        return this.cols;
    }

    public final Vector getCol(int col) {
        Vector res = new Vector(this.rows);
        for (int row = 1; row <= this.rows; ++row) {
            res.set(row, this.get(row, col));
        }
        return res;
    }

    public final Vector getRow(int row) {
        this.ensureIndicesOK(row, 1);
        return new Vector(this.fields[row - 1]);
    }

    public final void setCol(int col, Vector v) throws DimensionException {
        this.ensureCanBeColumn(v);
        for (int row = 1; row <= this.rows; ++row) {
            this.set(row, col, v.get(row));
        }
    }

    private void ensureCanBeColumn(Vector v) {
        if (v.getRows() != this.rows) {
            throw new DimensionException("Vector of size " + this.rows + " required.");
        }
    }

    public final void setRow(int row, Vector v) {
        this.ensureCanBeRow(v);
        for (int col = 1; col <= this.cols; ++col) {
            this.set(row, col, v.get(col));
        }
    }

    private void ensureCanBeRow(Vector v) {
        if (v.getRows() != this.cols) {
            throw new DimensionException("Vector of size " + this.cols + " required.");
        }
    }

    public Matrix cbind(Matrix other) {
        this.ensureMayBeCBinded(other);
        Matrix res = new Matrix(this.rows, this.cols + other.cols);
        res.copyWithWriteOffset(this.fields, 0, 0);
        res.copyWithWriteOffset(other.fields, 0, this.cols);
        return res;
    }

    public Matrix rbind(Matrix other) {
        this.ensureMayBeRBinded(other);
        Matrix res = new Matrix(this.rows, this.cols + other.cols);
        res.copyWithWriteOffset(this.fields, 0, 0);
        res.copyWithWriteOffset(other.fields, this.rows, 0);
        return res;
    }

    private void ensureMayBeCBinded(Matrix other) {
        if (other.getRows() != this.rows) {
            throw new DimensionException("Matrix with " + this.rows + " rows required.");
        }
    }

    private void ensureMayBeRBinded(Matrix other) {
        if (other.getCols() != this.cols) {
            throw new DimensionException("Matrix with " + this.rows + " rows required.");
        }
    }

    private void copyWithWriteOffset(double[][] fields, int rowOffset, int colOffset) {
        for (int i = 0; i < fields.length; ++i) {
            for (int j = 0; j < fields[i].length; ++j) {
                this.fields[i + rowOffset][j + colOffset] = fields[i][j];
            }
        }
    }

    public String toString() {
        StringBuilder str = new StringBuilder();
        for (int row = 1; row <= this.rows; ++row) {
            str.append("[ ");
            for (int col = 1; col < this.cols; ++col) {
                str.append(String.format("%.2f", this.get(row, col))).append(", ");
            }
            str.append(String.format("%.2f", this.get(row, this.cols))).append(" ]\n");
        }
        return str.toString();
    }

    public void invert() throws UninvertibleMatrixException {
        if (!this.isSquare()) {
            throw new UninvertibleMatrixException();
        }
        Matrix inverted = new Matrix(this.rows, this.cols);
        for (int col = 1; col <= this.cols; ++col) {
            int k = this.findRowUsedToDeleteOthers(col);
            if (k != col) {
                this.swapRows(k, col);
                inverted.swapRows(k, col);
            }
            this.deleteOtherRows(inverted, col);
            double d = 1.0 / this.get(col, col);
            this.multiplyRow(col, d);
            inverted.multiplyRow(col, d);
        }
        this.copy(inverted.fields);
    }

    private int findRowUsedToDeleteOthers(int col) throws UninvertibleMatrixException {
        for (int row = col; row <= this.rows; ++row) {
            if (!(Math.abs(this.get(row, col)) > 1.0E-5)) continue;
            return row;
        }
        throw new UninvertibleMatrixException();
    }

    private void deleteOtherRows(Matrix inverted, int col) {
        for (int row = 1; row <= this.rows; ++row) {
            if (row == col) continue;
            double coef = -this.get(row, col) / this.get(col, col);
            this.addRows(col, row, coef);
            inverted.addRows(col, row, coef);
        }
    }

    public void swapRows(int row1, int row2) {
        double[] aux = this.fields[row1 - 1];
        this.fields[row1 - 1] = this.fields[row2 - 1];
        this.fields[row2 - 1] = aux;
    }

    public Matrix getInverted() throws UninvertibleMatrixException {
        Matrix copy = new Matrix(this);
        copy.invert();
        return copy;
    }

    public void addRows(int from, int to, double coef) {
        for (int col = 1; col <= this.cols; ++col) {
            this.set(to, col, this.get(to, col) + coef * this.get(from, col));
        }
    }

    public void multiplyRow(int row, double coef) {
        for (int col = 1; col <= this.cols; ++col) {
            this.set(row, col, coef * this.get(row, col));
        }
    }
}

