{-# LANGUAGE RankNTypes #-}

-- deep embedding
data Expr a = One
            | Add (Expr a) (Expr a)
            | Var a
            | Let (Expr a) (a -> Expr a)

type ClosedExpr = forall a. Expr a

one = One
plus = Add

let_ :: Expr a -> (Expr a -> Expr a) -> Expr a
let_ e1 e2 = Let e1 (\x -> e2 (Var x))

-- preserving explicit sharing
eval :: Expr Int -> Int
eval One = 1
eval (Add e1 e2) = eval e1 + eval e2
eval (Var n) = n
eval (Let e1 e2) = let shared = eval e1 in eval (e2 shared)

treeI :: Int -> ClosedExpr
treeI 0 = one
treeI n = let shared = treeI (n-1) in shared `plus` shared

treeE :: Int -> ClosedExpr
treeE 0 = one
treeE n = let_ (treeE (n - 1)) (\shared -> shared `plus` shared)

-- explicit sharing now is also observable
text :: ClosedExpr -> String
text e = go e 0
  where
    go :: Expr String -> Int -> String
    go One _ = "1"
    go (Add e1 e2) c = "(" ++ go e1 c ++ " + " ++ go e2 c ++ ")"
    go (Var x) _ = x
    go (Let e1 e2) c =
        "(let " ++ v ++ " = " ++ go e1 (c + 1) ++
        " in " ++ go (e2 v) (c + 1) ++ ")"
      where
        v = "v" ++ show c

-- we could inline expression with explicit sharing
inline :: Expr (Expr a) -> Expr a
inline One = One
inline (Add e1 e2) = Add (inline e1) (inline e2)
inline (Var x) = x
inline (Let e1 e2) = inline (e2 (inline e1))

main = do
    putStrLn $ show $ eval $ one
    putStrLn $ show $ eval $ one `plus` one
    putStrLn $ show $ eval $ treeI 4
    putStrLn $ show $ eval $ treeE 22 -- pretty fast
    putStrLn $ show $ eval $ treeI 22 -- slow
    putStrLn $ show $ eval $ inline $ treeE 22 -- slow

